#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from httplib import HTTPMessage
from cStringIO import StringIO
import urllib

from zope.interface import implements

from twisted.internet.address import IPv4Address
from twisted.internet.defer import succeed, fail
from twisted.internet import interfaces
from twisted.web import resource, server

from imagestore.lib.tests import TestCase

from imagestore.lib.service import ServiceError
from imagestore.presenter import Presenter
from imagestore.resources import Root
from imagestore.model import ImageStoreResponse, ImageState
from imagestore.lib.tests import mocker


class DummyChannel:
    """Copied from twisted.web.test.test_web."""

    class TCP:
        port = 80

        def __init__(self):
            self.written = StringIO()

        def getPeer(self):
            return IPv4Address("TCP", '192.168.1.1', 12344)

        def write(self, bytes):
            assert isinstance(bytes, str)
            self.written.write(bytes)

        def writeSequence(self, iovec):
            map(self.write, iovec)

        def getHost(self):
            return IPv4Address("TCP", '10.0.0.1', self.port)

    class SSL(TCP):
        implements(interfaces.ISSLTransport)

    site = server.Site(resource.Resource())

    def __init__(self):
        self.transport = self.TCP()

    def requestDone(self, request):
        pass


class HTTPResponse(object):

    def __init__(self, data):
        sio = StringIO(data)
        self._status = sio.readline().split()
        self._message = HTTPMessage(sio)

    def getCode(self):
        return int(self._status[1])

    def getBody(self):
        self._message.rewindbody()
        return self._message.fp.read()

    def getHeader(self, name):
        return self._message.getheader(name)


class ResourcesTest(TestCase):

    def setUp(self):
        # We'll mock the methods we want, so we can pass None as
        # the task dispatcher.
        self.presenter = Presenter(None)

    def getRequest(self):
        channel = DummyChannel()
        channel.site = server.Site(Root(self.presenter))
        channel.transport.port = 52780
        request = server.Request(channel, 1)
        request.setHost("localhost", 52780)
        return request

    def getResponse(self, request):
        return HTTPResponse(request.transport.getvalue())

    def doRequest(self, method, path, body=""):
        request = self.getRequest()
        request.gotLength(len(body))
        if body:
            request.requestHeaders.addRawHeader(
                "content-type", "application/x-www-form-urlencoded")
            request.handleContentChunk(body)
        request.requestReceived(method, path, "HTTP/1.0")
        return self.getResponse(request)

    def _buildQuery(self, params):
        encodedParams = []
        for name, values in params.iteritems():
            name = name.replace("_", "-")
            if isinstance(values, list):
                for value in values:
                    encodedParams.append((urllib.quote(name),
                                          urllib.quote(value)))
            else:
                encodedParams.append((urllib.quote(name),
                                      urllib.quote(values)))
        return "&".join("%s=%s" % pair for pair in encodedParams)

    def GET(self, path, **params):
        if params:
            path = path + "?" + self._buildQuery(params)
        return self.doRequest("GET", path)

    def POST(self, path, **params):
        return self.doRequest("POST", path, self._buildQuery(params))

    def testDashboard(self):
        storeResponse = ImageStoreResponse({"Hello": "there!"})
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getDashboard()
        self.mocker.result(succeed(storeResponse))
        self.mocker.replay()

        httpResponse = self.GET("/api/dashboard")
        self.assertEquals(httpResponse.getBody(), '{"Hello": "there!"}')

    def testDashboardFailure(self):
        storeResponse = ImageStoreResponse({"Hello": "there!"})
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getDashboard()
        self.mocker.result(fail(ValueError()))
        self.mocker.replay()

        httpResponse = self.GET("/api/dashboard")
        self.assertEquals(httpResponse.getCode(), 200)
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": '
                          '"An error occurred in the image store proxy. '
                          'Please check the logs and report."}')

    def testDashboardServiceError(self):
        """ServiceError messages get passed as-is to the response."""
        storeResponse = ImageStoreResponse({"Hello": "there!"})
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getDashboard()
        self.mocker.result(fail(ServiceError("Some message")))
        self.mocker.replay()

        httpResponse = self.GET("/api/dashboard")
        self.assertEquals(httpResponse.getCode(), 200)
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": "Some message"}')

    def testSearch(self):
        storeResponse = ImageStoreResponse({"Hello": "there!"})
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.search("Something")
        self.mocker.result(succeed(storeResponse))
        self.mocker.replay()

        httpResponse = self.GET("/api/search", q="Something")
        self.assertEquals(httpResponse.getBody(), '{"Hello": "there!"}')

    def testStates(self):
        states = [ImageState({"image-uri": "http://uri1", "status": "s1"}),
                  ImageState({"image-uri": "http://uri2", "status": "s2"})]
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getStates(["uri1", "uri2"])
        self.mocker.result(succeed(states))
        self.mocker.replay()

        httpResponse = self.POST("/api/states", image_uri=["uri1", "uri2"])

        body = httpResponse.getBody()
        self.assertTrue(body.startswith('{"states":'), body)
        storeResponse = ImageStoreResponse(body)
        self.assertEquals(storeResponse["states"], states)

    def testStatesWithoutUris(self):
        storeResponse = ImageStoreResponse({"Hello": "there!"})
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getStates([])
        self.mocker.result(succeed(storeResponse))
        self.mocker.replay()

        httpResponse = self.POST("/api/states")

    def testInstallWithoutClientId(self):
        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/install")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message":'
                          ' "Bad request signature (ClientId not provided)"}')

    def testInstallWithUnknownClientId(self):
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed(None))
        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/install",
                                 ClientId="admin")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": "Bad request signature '
                          '(unknown ClientId)"}')

    def testInstallWithMissingSignature(self):
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))
        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/install",
                                 ClientId="admin")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": "Bad request signature"}')

    def testInstallWithBadSignature(self):
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))
        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/install",
                                 ClientId="admin",
                                 Signature="L2wrVB4QzgFGeJjqe6vLDcR7ku"
                                           "FlrpjqkgJ0rGS0bdM=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": "Bad request signature"}')

    def testInstall(self):
        presenterMock = self.mocker.patch(self.presenter)

        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))

        presenterMock.startInstall("http://uri")
        self.mocker.result(succeed("!UNUSED!"))

        states = [ImageState({"image-uri": "http://uri1", "status": "s1"})]
        presenterMock.getStates(["http://uri"])
        self.mocker.result(succeed(states))

        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/install",
                                 ClientId="admin",
                                 Signature="l2wrVB4QzgFGeJjqe6vLDcR7ku"
                                           "FlrpjqkgJ0rGS0bdM=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")

        body = httpResponse.getBody()
        self.assertTrue(body.startswith('{"states":'), body)
        storeResponse = ImageStoreResponse(body)
        self.assertEquals(storeResponse["states"], states)

    def testInstallError(self):
        """
        On errors, since the action really happens in background, the
        image states should be returned anyway.  Any errors will be
        inserted in the state itself and returned on this or on future
        state requests.
        """
        presenterMock = self.mocker.patch(self.presenter)

        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))

        presenterMock.startInstall("http://uri")
        self.mocker.result(fail(ValueError("!IGNORED!")))

        states = [ImageState({"image-uri": "http://uri1", "status": "s1"})]
        presenterMock.getStates(["http://uri"])
        self.mocker.result(succeed(states))

        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/install",
                                 ClientId="admin",
                                 Signature="l2wrVB4QzgFGeJjqe6vLDcR7ku"
                                           "FlrpjqkgJ0rGS0bdM=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")

        body = httpResponse.getBody()
        self.assertStartsWith(body, '{"states":')
        storeResponse = ImageStoreResponse(body)
        self.assertEquals(storeResponse["states"], states)

    def testCancel(self):
        presenterMock = self.mocker.patch(self.presenter)

        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))

        presenterMock.cancelChanges("http://uri")
        self.mocker.result(succeed("!UNUSED!"))

        states = [ImageState({"image-uri": "http://uri1", "status": "s1"})]
        presenterMock.getStates(["http://uri"])
        self.mocker.result(succeed(states))

        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/cancel",
                                 ClientId="admin",
                                 Signature="/EhSldULIJrVYLZ4IuiCXbzlA1X"
                                           "wQg8gmq7xkyyAdDU=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")

        body = httpResponse.getBody()
        self.assertTrue(body.startswith('{"states":'), body)
        storeResponse = ImageStoreResponse(body)
        self.assertEquals(storeResponse["states"], states)

    def testCancelWithBadSignature(self):
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))
        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/cancel",
                                 ClientId="admin",
                                 Signature="/ehSldULIJrVYLZ4IuiCXbzlA1X"
                                           "wQg8gmq7xkyyAdDU=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": "Bad request signature"}')

    def testClearError(self):
        presenterMock = self.mocker.patch(self.presenter)

        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))

        presenterMock.clearError("http://uri")
        self.mocker.result(succeed("!UNUSED!"))

        states = [ImageState({"image-uri": "http://uri1", "status": "s1"})]
        presenterMock.getStates(["http://uri"])
        self.mocker.result(succeed(states))

        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/clear-error",
                                 ClientId="admin",
                                 Signature="LVfe8vi7uDCE4/MU04xwMIB3tw9Su/"
                                           "+hX6jLIK0vSPI=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")

        body = httpResponse.getBody()
        self.assertTrue(body.startswith('{"states":'), body)
        storeResponse = ImageStoreResponse(body)
        self.assertEquals(storeResponse["states"], states)

    def testImageUriEncodedWithUrlSafeBase64(self):
        presenterMock = self.mocker.patch(self.presenter)

        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))

        presenterMock.startInstall("\xd7\xf9")
        self.mocker.result(succeed("!UNUSED!"))

        states = [ImageState({"image-uri": "http://uri1", "status": "s1"})]
        presenterMock.getStates(["\xd7\xf9"])
        self.mocker.result(succeed(states))

        self.mocker.replay()

        httpResponse = self.POST("/api/images/1_k=/install",
                                 ClientId="admin",
                                 Signature="/YXMMEIe0x1bb337KzVZgFowTn"
                                           "PIKQYuBJxysdadKUk=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")

        body = httpResponse.getBody()
        self.assertTrue(body.startswith('{"states":'), body)
        storeResponse = ImageStoreResponse(body)
        self.assertEquals(storeResponse["states"], states)

    def testClearErrorWithBadSignature(self):
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getSecretKey("admin")
        self.mocker.result(succeed("secret"))
        self.mocker.replay()

        httpResponse = self.POST("/api/images/aHR0cDovL3VyaQ==/clear-error",
                                 ClientId="admin",
                                 Signature="lVfe8vi7uDCE4/MU04xwMIB3tw9Su/"
                                           "+hX6jLIK0vSPI=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": "Bad request signature"}')

    def testImageUriNotImplemented(self):
        httpResponse = self.GET("/api/images/aHR0cDovL3VyaQ==")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message":'
                          ' "Individual image data not yet provided by'
                          ' the proxy API"}')

    def testGetSecretKeyFailure(self):
        presenterMock = self.mocker.patch(self.presenter)
        presenterMock.getSecretKey("admin")
        self.mocker.result(fail(ValueError("BOOM!")))
        self.mocker.replay()

        httpResponse = self.POST("/api/images/1_k=/install",
                                 ClientId="admin",
                                 Signature="/YXMMEIe0x1bb337KzVZgFowTn"
                                           "PIKQYuBJxysdadKUk=",
                                 SignatureMethod="HmacSHA256",
                                 SignatureVersion="2")
        self.assertEquals(httpResponse.getBody(),
                          '{"error-message": '
                          '"Proxy failed to retrieve Eucalyptus credentials"}')
