/*
 * Copyright 2021-2024 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef IA_LOG_H_
#define IA_LOG_H_

#include "ia_types.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef __func__
/*LDRA_INSPECTED 219 S */
/*LDRA_INSPECTED 86 S */
/*LDRA_INSPECTED 626 S */
#define __func__ __FUNCTION__
#endif

/*!
 * \brief Macro for not logging.
 */
#define IA_NOLOG(...) (CAST_TO_TYPE(void, 0)) /* LDRA_INSPECTED 58 S */
/*!
 * \brief Macro wrapper for ia_log function.
 */
#define IA_LOG(level, fmt, ...) (CAST_TO_TYPE(void, ia_log((level), (fmt), ## __VA_ARGS__)))

/*!
 * \brief Macro wrapper for ia_log_with_tag function.
 */
#define IA_LOG_TAG(tag, level, fmt, ...) (CAST_TO_TYPE(void, ia_log_with_tag((tag), (level), (fmt), ## __VA_ARGS__)))

/*!
 * \brief Logging levels.
 */
typedef enum
{
    ia_log_error,
    ia_log_debug,
    ia_log_info,
    ia_log_perf,
} ia_log_level;

/*!
 * \brief Initializes ia_log library with external function pointers for logging.
 * If this function is not called, logging will be directed to stdout and stderr.
 *
 * \param[in] a_env_ptr Structure containing the addresses to logging functions.
 *                      If NULL is passed as function pointer, there won't be log prints.
 * \return              Error code.
 */
LIBEXPORT ia_err
ia_log_init(ia_env *a_env_ptr);

/*!
 * \brief Prints formatted string.
 *
 *
 * \param[in] level Log level.
 * \param[in] fmt   Format string.
 * \param[in] ...   Variables to be printed out as defined by the format string.
 */

/*LDRA_INSPECTED 306 S */
LIBEXPORT void
ia_log(ia_log_level level, const char *fmt, ...);

/*!
 * \brief Prints formatted string with log tag
 *
 *
 * \param[in] print_tag   Log level.
 * \param[in] level       Log level.
 * \param[in] fmt         Format string.
 * \param[in] ...         Variables to be printed out as defined by the format string.
 */

/*LDRA_INSPECTED 306 S */
LIBEXPORT void
ia_log_with_tag(const char *print_tag, ia_log_level level, const char *fmt, ...);

/*!
 * \brief Prints formatted string with interpreted error code.
 *
 *
 * \param[in]     error               Error code.
 * \return                            Pointer to all Error string.
 */
LIBEXPORT const char *
ia_log_strerror(ia_err error);

/*!
 * \brief De-initialization of ia_log library.
 * Must be called if ia_log_init() was called.
 */
LIBEXPORT void
ia_log_deinit(void);

#ifdef __cplusplus
}
#endif

#endif /* _IA_LOG_H_ */
