
/*
 ***********************************************************************
 * Copyright 2009-2010 by Freescale Semiconductor, Inc.
 * All modifications are confidential and proprietary information
 * of Freescale Semiconductor, Inc. ALL RIGHTS RESERVED.
 ***********************************************************************
 */
 
/**
 * @file avi_parser_api.h
 * @AVI file parser header files
 */

#ifndef FSL_AVI_PARSER_API_HEADER_INCLUDED
#define FSL_AVI_PARSER_API_HEADER_INCLUDED

/* common files of mmalyer */
#include "common/fsl_types.h"
#include "common/file_stream.h"
#include "common/fsl_media_types.h"
#include "common/fsl_parser.h"


/***************************************************************************************
*                  Data Structures & Constants
***************************************************************************************/
#define MAX_AVI_TRACKS 24 /* maximum media tracks to support for an AVI movie. 
                                                If there are more tracks, those with a larger track number will be overlooked */

/* 
 * Flags of a sample.
 * AVI own sample flags, for API AviGetNextSample(). Can only use high 16 bits.
 * Lower 16 bits is reserved for common sample flags of all parsers.
 */
#define FLAG_UNCOMPRESSED_SAMPLE 0X0100 /* This is a uncompressed sample. 
                                        Warning: 
                                        Video track may have both compressed & uncompressed samples.
                                        But some AVI clips seem to abuse this flag, sync samples are mark as uncompressed, 
                                        although they are actually compressed ones.
                                        Now suggest not care this flag.*/


/*
 * error code of the AVI parser API. 
 
 */

typedef enum
{
    AVI_ERR_WRONG_FILE_SIZE = -111,
    

    AVI_ERR_NOT_AVI_FILE = -120, /* This is not an AVI file. No RIFF AVI header found */
    AVI_ERR_UNKNOWN_STREAM_FORMAT = -121,
    AVI_ERR_WRONG_MEDIA_TYPE = -122, /* An API is called on a track of wrong media type. */
    AVI_ERR_WRONG_VIDEO_FORMAT = -123,
    AVI_ERR_WRONG_AUDIO_FORMAT = -124,
    AVI_ERR_INVALID_ATOM = -125,
    
    AVI_ERR_CORRUPTED_INDEX = -130,
    AVI_ERR_WRONG_INDEX_SAMPLE_SIZE = -131,  /* the index give wrong sample size */
    AVI_ERR_WRONG_INDEX_SAMPLE_OFFSET = -132, /* the index give wrong sample offset */
    AVI_ERR_WRONG_IDX1_LIST_SIZE = -133,    
    AVI_ERR_WRONG_AVI2_INDEX_SIZE = -134,
    AVI_ERR_WRONG_AVI2_INDEX_ENTRY_SIZE = -135,
    AVI_ERR_INDEX_TYPE_NOT_SUPPORTED = -136,
    AVI_ERR_SUPER_INDEX_ENTRY_NOT_FOUND = -137,
    AVI_ERR_EMPTY_INDEX = -138,
    
    
    AVI_ERR_DRM_NOT_PROTECTED = -142, /* call DRM APIs for a not protected clip */
    AVI_ERR_DRM_INVALID_CONTEXT = -143, /* DRM context is invalid */
    AVI_ERR_DRM_PREV_PLAY_NOT_CLEAERED = -144,
    AVI_ERR_DRM_INVALID_CALLBACK = -145,
    AVI_ERR_DRM_OTHERS = -146, /* Reserved for other DRM errors.*/
    AVI_ERR_WRONG_DRM_INFO_SIZE = -148,

    AVI_ERR_ZERO_STREAM_RATE = -150, /* The rate of a track is ZERO. Can not calculate time stamp.*/


    AVI_ERR_NO_INDEX = -152, /* There is no index, but not affect normal playback */    
    AVI_ERR_INDEX_ALREADY_LOADED = -153, /* the index table is already loaded or imported */
    AVI_ERR_NO_SYNC_SAMPLE_IN_INDEX = -154, /* No sync sample found in the index table */

    AVI_ERR_NO_USER_DATA = -155, /* the user data is not available */

    AVI_ERR_WRONG_CHUNK = -160, /* bad chunk id found */    
    AVI_ERR_WRONG_CHUNK_SIZE = -161, /*Invalid chunk size, greater than the tatal movie size or maximum sample size of the track.*/
    AVI_ERR_WRONG_CHUNK_TAG = -162,

    AVI_ERR_WRONG_TRACK_NUM = -163,

    AVI_ERR_WRONG_MOVIE_LIST_SIZE = -170,
    AVI_ERR_NO_MOVIE_LIST = -171,

    AVI_ERR_NO_PRIMARY_TRACK = -180, /* Neither video nor audio available as the primary track */

    AVI_ERR_INVALID_STREAM_NAME = -190
     
}AVI_PARSER_ERROR_CODE;



/*
 * Handle of the AVI core parser created. 
 */
typedef void * AviParserHandle;


/* 
 * Avi parser memory callback funtion pointer table.An "context" is used for the wrapper usage.
 */
typedef struct 
{  
    void* (*Calloc) (uint32 numElements, uint32 size);
    void* (*Malloc) (uint32 size);
    void  (*Free) (void * ptr);     
    void* (*ReAlloc)(void * ptr, uint32 size); /* necessary for index scanning!*/

}AviMemoryOps; /* callback operation callback table */




typedef struct
{
    int32 (*WriteDrmFragment)(uint32 fragmentNum, uint8 * data, uint32 dataLength );
    int32 (*ReadDrmFragment)(uint32 fragmentNum, uint8 * data, uint32 dataLength ); 
}AviDrmOps;     
/* DRM callback funtion, to read/write local drm information. Application decide where to hide the information.*/

/* return value of the DRM callback function, consist with DRM lib */
#define AVI_DRM_LOCAL_SUCCESS                  0
#define AVI_DRM_ERROR_READING_MEMORY    12			
#define AVI_DRM_ERROR_WRITING_MEMORY    13




/***************************************************************************************
*                  API Funtions
* For calling sequence, please refer to the end of this file.
***************************************************************************************/
#ifdef __cplusplus
#define EXTERN extern "C"
#else
#define EXTERN 
#endif

/**
 * function to get the AVI core parser version.
 *  
 * @return Version string.
 */
EXTERN const char * AviParserVersionInfo();


/**
 * function to create the AVI core parser.
 * 
 * @param stream [in]   Source stream of the AVI file.
 *                      It implements functions to open, close, tell, read and seek a file.
 *
 * @param memOps [in]   Memory operation callback table.
 *                      It implements the functions to malloc, calloc, realloc and free memory.
 *
 * @param drmOps [in]   DRM callback function table.
 *                      It implements the functions to read and write local DRM memory.
 *
 * @param context [in]  Wrapper context for the callback functions. Avi parser never modify it.
 * @param parserHandle [out] Handle of the AVI core parser if succeeds. NULL for failure.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE  AviCreateParser(file_stream_t * stream,                                                        
                                                        AviMemoryOps * memOps,
                                                        AviDrmOps * drmOps,
                                                        void * context,
                                                        AviParserHandle * parserHandle);

/**
 * function to delete the AVI core parser.
 *
 * @param parserHandle Handle of the AVI core parser.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviDeleteParser(AviParserHandle parserHandle);

/**
 * function to parse the AVI file header. It shall be called after AVI parser is created 
 * and it will probe whether the movie can be handled by this AVI parser.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviParseHeader(AviParserHandle parserHandle);


/***************************************************************************************
*                  DRM APIs Begin
***************************************************************************************/

/**
 * DRM interface.function to see whether file is protected by DRM. 
 * The wrapper shall call the DRM interface right after the file header is parsed for a quick decision. 
 * before doing the time-consuming task such as initialize index table.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param isProtected [out]True for protected file.
 */
EXTERN AVI_PARSER_ERROR_CODE AviIsProtected(AviParserHandle parserHandle, bool * isProtected);

/**
 * DRM interface.function to see whether file is a rental or purchased movie.
 * This API shall be called once before playing a protected clip.
 *
 * @param parserHandle[in] Handle of the avi core parser.
 * @param isRental[out] True for a rental file and False for a puchase file. Reatanl file has a view limit.
 * @param viewLimit[out] View limit if a rental file.
 * @param viewCount [out]Count of views played already.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviQueryContentUsage( AviParserHandle parserHandle,
                                            bool * isRental, 
                                            uint32 * viewLimit, 
                                            uint32 * viewCount);


/**
 * DRM interface. function to check the video output protection flag.
 *
 * @param parserHandle[in] - Handle of the AVI core parser.
 * @param cgmsaSignal[out] - 0, 1, 2, or 3 based on standard CGMSA signaling.
 * @param acptbSignal[out] - 0, 1, 2, or 3 based on standard trigger bit signaling.
 *                                      acptb values:
 *                                      0 = Off.
 *                                      1 = Auto gain control / pseudo sync pulse.
 *                                      2 = Two line color burst.
 *                                      3 = Four line color burst.
 * @param[out] digitalProtectionSignal - 0=off, 1=on.
 * @return PARSER_SUCCESS - success. Others - failure.
 */

EXTERN AVI_PARSER_ERROR_CODE AviQueryOutputProtectionFlag( AviParserHandle parserHandle, 
                                                          uint8 * cgmsaSignal,
                                                          uint8 * acptbSignal,
                                                          uint8 * digitalProtectionSignal);

/**
 * DRM interface.function to commit playing the protected file.The wrapper shall call it before playback is started.
 *
 * @param parserHandle[in] Handle of the AVI core parser.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviCommitPlayback(AviParserHandle parserHandle);

/**
 * DRM interface.function to end playing the protected file. 
 * The wrapper shall call it after playback is stopped.
 * Otherwise error "AVI_ERR_DRM_PREV_PLAY_NOT_CLEAERED" on next playback.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviFinalizePlayback(AviParserHandle parserHandle);

/***************************************************************************************
*                  DRM APIs End
***************************************************************************************/


/**
 * Function to intialize the index table by scanning the index in the file. 
 * For a movie played for the 1st time, the index table has to be loaded from the file. 
 * The index table increases with the movie duration. So the longer the movie is, 
 * the more time it takes to load the index.
 *
 * Seeking and trick mode can be performed on a movie only if it has a index. 
 * But normal playback does not depend on the index. So even if this function fails, 
 * normal playback can still work as long as movie data are right.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviInitializeIndex(AviParserHandle parserHandle);

/**
 * function to import the index table of a track from outside, instead of scanning the file. 
 * This can reduce time to open a 2nd-play movie if its index table has been exported on 1st play.
 * To save the memory used, import/export a track's index per time.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param buffer [in] Buffer containing the index data.
 * @param size [in] Size of the index data in the buffer, in bytes.
 * @return
 */ 
EXTERN AVI_PARSER_ERROR_CODE AviImportIndex(AviParserHandle parserHandle, uint32 trackNum, uint8 * buffer, uint32 size);

/**
 * function to export the index table of track from outside, after the index table is loaded by scanning the file at 1st play.
 * This function is usually used on a 1st play movie.
 * This can reduce time to open a 2nd-play movie if its index table has been exported on 1st play.
 * To save the memory used, import/export a track's index per time.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param buffer [in] Buffer to export the index data. 
 *                             If this parameter is NULL, just return the size of buffer needed without exporting the index data.
 *
 * @param size [in/out] Size of the the buffer as input, in bytes.
 *                                 Size of the index data in the buffer as output, in bytes.                       
 * @return
 */ 
EXTERN AVI_PARSER_ERROR_CODE AviExportIndex(AviParserHandle parserHandle, uint32 trackNum, uint8 * buffer, uint32 *size);

/**
 * function to tell whether the movie is seekable. A seekable AVI movie must have the index table.
 * If the file's index table is loaded from file or imported successfully, it's seekable.
 * Seeking and trick mode can be performed on a seekable file.
 * If the index table is corrupted, the file is NOT seekable. This function will fail and return value can tell the error type. 
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param seekable [out] true for seekable and false for non-seekable 
 * @return 
 */ 
EXTERN AVI_PARSER_ERROR_CODE AviIsSeekable(AviParserHandle parserHandle, bool * seekable);

/**
 * function to tell how many tracks in the movie. 
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param numTracks [out] Number of tracks.
 * @return 
 */ 
EXTERN AVI_PARSER_ERROR_CODE AviGetNumTracks(AviParserHandle parserHandle, uint32 * numTracks);



/**
 * function to tell the user data information (title, artist, genre etc) of the movie. User data API.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param id [in] Type of the user data.
 * @param buffer [out] Buffer containing the information. The core parser manages this buffer and the user shall NOT free it.
 *                               If no such info is availabe, this value will be set to NULL.
 * @param size [out] Length of the information in bytes. The informaiton is usually a null-terminated ASCII string.
 *                               If no such info is available, this value will be set to 0.
 * @return
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetUserData(AviParserHandle parserHandle, UserDataID id, uint8 ** buffer, uint32 *size);



/**
 * function to tell the movie duration. 
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param usDuration [out] Duration in us.
 * @return 
 */ 
EXTERN AVI_PARSER_ERROR_CODE AviGetMovieDuration(AviParserHandle parserHandle,  uint64 * usDuration);


/**
 * function to tell a track's duration. 
 * The tracks may have different durations. 
 * And the movie's duration is usually the video track's duration (maybe not the longest one).
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param usDuration [out] Duration in us.
 * @return 
 */ 
EXTERN AVI_PARSER_ERROR_CODE AviGetTrackDuration(AviParserHandle parserHandle, uint32 trackNum, uint64 * usDuration);


/**
 * function to tell the type of a track. 
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 *
 * @param mediaType [out] Media type of the track. (video, audio, subtitle...)
 *                        "MEDIA_TYPE_UNKNOWN" means the media type is unknown.
 *
 * @param decoderType [out] Decoder type of the track if available. (eg. MPEG-4, H264, AAC, MP3, AMR ...)
 *                          "UNKNOWN_CODEC_TYPE" means the decoder type is unknown.
 *
 * @param decoderSubtype [out] Decoder Subtype type of the track if available. (eg. AMR-NB, AMR-WB ...)
 *                             "UNKNOWN_CODEC_SUBTYPE" means the decoder subtype is unknown.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetTrackType(   AviParserHandle parserHandle, 
                                                uint32 trackNum, 
                                                uint32 * mediaType,
                                                uint32 * decoderType,
                                                uint32 * decoderSubtype);

/**
 * function to tell the codec specific information of a track.
 * It's the data of stream format atom (strf).
 * It's a Windows bitmap header for video track (at least 40 bytes)
 * and a Windows Waveform audio header for audio (at least 16 or 18 bytes).
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param data [out] Buffer holding the codec specific information. The user shall never free this buffer.
 * @param size [out] Size of the codec specific information, in bytes.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetCodecSpecificInfo(AviParserHandle parserHandle, 
                                                uint32 trackNum, 
                                                uint8 ** data,
                                                uint32 * size);

/**
 * function to tell the maximum sample size of a track. 
 * Avi parser read A/V tracks sample by sample. The max sample size can help the user to prepare a big enough buffer.
 * Warning!The "max sample size" can be zero if the file header information is not complete or index table is not available.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param size [out] Max sample size of the track. Warning!It can be zero if index table does not exist.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetMaxSampleSize(AviParserHandle parserHandle, uint32 trackNum, uint32 * size);

/**
 * function to tell the language of a track used.
 * This is helpful to select an video/audio/subtitle track or menu pages.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param threeCharCode [out] Three or two character language code. 
 *                  See ISO 639-2/T for the set of three character codes.Eg. 'eng' for English.
 *                  Four special case:
 *                  mis- "uncoded languages"
 *                  mul- "multiple languages"
 *                  und- "undetermined language"
 *                  zxx- "no linguistic content"
 *                  
 *                  See ISO 639 for the set of two character codes. Eg. 'en' for English and'zh'for chinese.
 *                  If ISO 639 is used, the 3rd character will be '\0'.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetLanguage(AviParserHandle parserHandle, uint32 trackNum, uint8 *threeCharCode);


/* NOT implemented */
EXTERN AVI_PARSER_ERROR_CODE AviGetScale(AviParserHandle parserHandle, uint32 trackNum, uint32 *scale);
/* NOT implemented */
EXTERN AVI_PARSER_ERROR_CODE AviGetRate(AviParserHandle parserHandle, uint32 trackNum, uint32 *rate);
/* NOT implemented */
EXTERN AVI_PARSER_ERROR_CODE AviGetStartDelay(AviParserHandle parserHandle, uint32 trackNum, uint32 *usStartDelay);

/**
 * function to tell the bitrate of a track.
 * For CBR stream, the real bitrate is given.
 * For VBR stream, 0 is given since the bitrate varies during the playback and AVI parser does not 
 * calculate the peak or average bit rate.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based.
 * @param bitrate [out] Bitrate. For CBR stream, this is the real bitrate. 
 *                                            For VBR stream, the bitrate is 0 since max bitrate is usually not available.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetBitRate(AviParserHandle parserHandle, uint32 trackNum, uint32 *bitrate);


/**
 * function to tell the sample duration in us of a track. 
 * If the sample duration is not a constant (eg. some audio, subtilte), 0 is given.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. 
 * @param usDuration [out] Sample duration in us. If sample duration is not a constant, this value is 0.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetSampleDuration(AviParserHandle parserHandle, uint32 trackNum, uint64 *usDuration);


/**
 * function to tell the width in pixels of a video track. 
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a video track.
 * @param width [out] Width in pixels.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetVideoFrameWidth(AviParserHandle parserHandle, uint32 trackNum, uint32 *width) ;

/**
 * function to tell the height in pixels of a video track. 
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a video track.
 * @param height [out] Height in pixels.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetVideoFrameHeight(AviParserHandle parserHandle, uint32 trackNum, uint32 *height);


/**
 * function to tell how many channels in an audio track.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to an audio track.
 * @param numchannels [out] Number of the channels. 1 mono, 2 stereo, or more for multiple channels.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetAudioNumChannels(AviParserHandle parserHandle, uint32 trackNum, uint32 *numchannels);

/**
 * function to tell the audio sample rate (sampling frequency) of an audio track.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to an audio track.
 * @param sampleRate [out] Audio integer sample rate (sampling frequency).
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetAudioSampleRate(AviParserHandle parserHandle, uint32 trackNum, uint32 * sampleRate);



/**
 * function to tell the bits per sample for a PCM audio track.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a PCM audio track.
 * @param bitsPerSample [out] Bits per PCM sample.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetAudioBitsPerSample(AviParserHandle parserHandle, uint32 trackNum, uint32 * bitsPerSample);

/**
 * Function to tell the width of a text track.
 * The text track defines a window to display the subtitles. 
 * This window shall be positioned in the middle of the screen. 
 * And the sample is displayed in the window.How to position the sample within the window is defined by the sample data.
 * The origin of window is always (0, 0).
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a text track.
 * @param width [out] Width of the text track, in pixels.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetTextTrackWidth(AviParserHandle parserHandle, 
                                                uint32 trackNum, 
                                                uint32 * width);
                                                
/**
 * Function to tell the height of a text track.
 * The text track defines a window to display the subtitles. 
 * This window shall be positioned in the middle of the screen. 
 * And the sample is displayed in the window.How to position the sample within the window is defined by the sample data.
 * The origin of window is always (0, 0).
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a text track.
 * @param height [out] Height of the window, in pixels.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetTextTrackHeight(AviParserHandle parserHandle, 
                                                uint32 trackNum,
                                                uint32 * height);


/**
 * Optional API. Function to tell the next sample size of a track. 
 * This function shall be called when current sample output is finished. Otherwise
 * it will give the size of current sample.
 * 
 * Warning: if the sample size is an ODD number, there is one padding byte will be read
 * by the parser after the sample data. The user shall prepare an buffer enough to 
 * hold the padding byte. If the sample size is EVEN, no such need.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a text track.
 * @param sampleSize [out] Size of the next sample, in bytes.
 * @return 
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetNextSampleSize( AviParserHandle parserHandle, 
                                            uint32 trackNum,                                           
                                            uint32 * sampleSize);

/**
 * function to read the next sample from a track.
 *
 * For clips not protected by DRM, it supports partial output of large samples:
 * If the entire sample can not be output by calling this function once, its remaining data  
 * can be got by repetitive calling the same function.
 *
 * For DRM-protected clips, can only output entire samples for decryption need.
 *
 * For A/V tracks, the time stamp of samples are continuous. If a sample is output, its start time and end time 
 * are also output.
 * But for subtitle text tracks, the time stamp & duration are discontinuous and encoded in the sample data.So the parser 
 * gives an "estimated" time stamp. The decoder shall decode the accurate time stamp.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track to read, 0-based.
 * @param sampleData [in]   Buffer to hold the sample data. 
 *                          For DRM-protected clips, it must be large enough to hold the entire sample for decryption need. 
 *                          Otherwise this function will fail.
 *                          For non-protected clips, if the buffer is not big enough, only the 1st part of sample is output.
 *
 * @param dataSize [in/out]  Size of the buffer as input, in bytes.
 *                             As output:
 *
 *                              If a sample or part of sample is output successfully (return value is PARSER_SUCCESS ), 
 *                              it's the size of the data actually got.
 *
 *                              If the sample can not be output at all because buffer is too small 
 *                              (the return value is PARSER_INSUFFICIENT_MEMORY), it's the buffer size needed. 
 *                              Only for DRM-protected files.
 *
 * @param usStartTime [out] Start time of the sample in us (timestamp)
 * @param usEndTime [out] End time of the sample in us.
 *
 * @param flag [out] Flags of this sample if a sample is got successfully.
 *
 *                            FLAG_SYNC_SAMPLE    
 *                                  Whether this sample is a sync sample (key frame). 
 *                                  For non-video media, the wrapper shall take every sample as sync sample.
 *
 *                            FLAG_UNCOMPRESSED_SAMPLE
 *                                  Whethter this sample is a uncompressed one. Uncompressed samples shall bypass the decoder.
 *                                  Warning: Video track may have both compressed & uncompressed samples.
 *                                                But some AVI clips seem to abuse this flag, sync samples are mark as uncompressed, 
 *                                                  although they are actually compressed ones.
 *                            
 *                            FLAG_SAMPLE_ERR_CONCEALED
 *                                  There is error in bitstream but a sample is still got by error concealment.                                  
 *                            
 *                            FLAG_SAMPLE_SUGGEST_SEEK
 *                                  A seeking on ALL tracks is suggested although samples can be got by error concealment. 
 *                                  Because there are many corrupts, 
 *                                  and A/V sync is likely impacted by simple concealment(scanning bitstream).
 *                          
 *                            FLAG_SAMPLE_NOT_FINISHED
 *                                  Sample data output is not finished because the buffer is not big enough.                                 
 *                                  Need to get the remaining data by repetitive calling this func.
 *                                  This feature is only for non-protected clips.
 *
 *
 * @return  PARSER_SUCCESS     An entire sample or part of it is got successfully.
 *          PARSER_EOS     No sample is got because of end of the track.
 *          PARSER_INSUFFICIENT_MEMORY Buffer is not big enough to hold the entire sample.                                    
 *                                  The user can allocate a larger buffer and call this API again. (Only for DRM-protected clips).
 *          PARSER_READ_ERROR  File reading error. No need for further error concealment.
 *          PARSER_ERR_CONCEAL_FAIL  There is error in bitstream, and no sample can be found by error concealment. A seeking is helpful.
 *          Others ...
 */
EXTERN AVI_PARSER_ERROR_CODE AviGetNextSample( AviParserHandle parserHandle, 
                                        uint32 trackNum,
                                        uint8 * sampleData,  
                                        uint32 * dataSize, 
                                        uint64 * usStartTime, 
                                        uint64 * usEndTime,
                                        uint32 * flag);


/**
 * Function to seek a track to a target time. It will seek to a sync sample of the time stamp 
 * matching the target time. Due to the scarcity of the video sync samples (key frames), 
 * there can be a gap between the target time and the timestamp of the matched sync sample. 
 * So this time stamp will be output to as the accurate start time of the following playback segment.
 * NOTE: Seeking to the beginning of the movie (target time is 0 us) does not require the index table.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track to seek, 0-based.
 * @param usTime [in/out]  Target time to seek as input, in us. 
 *                         Actual seeking time, timestamp of the matched sync sample, as output.
 *
 * @param flag [in]  Control flags to seek.
 *
 *                      SEEK_FLAG_NEAREST   
 *                      Default flag.The matched time stamp shall be the nearest one to the target time (may be later or earlier).
 *                           
 *                      SEEK_FLAG_NO_LATER  
 *                      The matched time stamp shall be no later than the target time.
 *                                                              
 *                      SEEK_FLAG_NO_EARLIER
 *                      The matched time stamp shall be no earlier than the target time.
 *
 * @return  PARSER_SUCCESS    Seeking succeeds.
 *          PARSER_ERR_NOT_SEEKABLE    Seeking fails because he movie is not seekable (index not available)            
 *          Others      Seeking fails for other reason.
 */
EXTERN  AVI_PARSER_ERROR_CODE AviSeek(AviParserHandle parserHandle,
                                     uint32 trackNum,
                                     uint64 * usTime,
                                     uint32 flag);


/**
 * Optional API. Function to tell the size of next or previous sync sample of a track. 
 * This function shall be called when current sample output is finished. Otherwise
 * it will give the size of current sample.
 *
 * Warning: if the sample size is an ODD number, there is one padding byte will be read
 * by the parser after the sample data. The user shall prepare an buffer enough to 
 * hold the padding byte. If the sample size is EVEN, no such need.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track, 0-based. It must point to a text track.
 * @param direction [in]  Direction to get the sync sample.
 *                           FLAG_FORWARD   To get the next sync sample from current reading position.
 *                           FLAG_BACKWARD  To get the previous sync sample from current reading position.
 * @param sampleSize [out] Size of the sync sample, in bytes.
 * @return 
 */
EXTERN  AVI_PARSER_ERROR_CODE AviGetNextSyncSampleSize( AviParserHandle parserHandle, 
                                            uint32 trackNum,
                                            uint32 direction,
                                            uint32 * sampleSize);

/**
 * function to get the next or previous sync sample (key frame) from current reading position of a track.
 * For trick mode FF/RW.
 * Also support partial output of large samples for clips not protected by DRM.
 * If not the entire sample is got, its remaining data can be got by repetitive calling the same function.
 *
 * @param parserHandle [in] Handle of the AVI core parser.
 * @param trackNum [in] Number of the track to read, 0-based. 
 * @param direction [in]  Direction to get the sync sample.
 *                           FLAG_FORWARD   Read the next sync sample from current reading position.
 *                           FLAG_BACKWARD  Read the previous sync sample from current reading position.
 *
 * @param sampleData [in]   Buffer to hold the sample data. 
 *                          For DRM-protected clips, it must be large enough to hold the entire sample for decryption need. 
 *                          Otherwise this function will fail.
 *                          For non-protected clips, if the buffer is not big enough, only the 1st part of sample is output.
 *
 * @param dataSize [in/out]  Size of the buffer as input, in bytes.
 *                             As output:
 *
 *                              If a sample or part of sample is output successfully (return value is PARSER_SUCCESS ), 
 *                              it's the size of the data actually got.
 *
 *                              If the sample can not be output at all because buffer is too small 
 *                              (the return value is PARSER_INSUFFICIENT_MEMORY), it's the buffer size needed. 
 *                              Only for DRM-protected files.
 *
 * @param usStartTime [out] Start time of the sample in us (timestamp)
 * @param usEndTime [out] End time of the sample in us.
 * @param flag [out] Flags of this sample.
 *                            FLAG_SYNC_SAMPLE    
 *                                  Whether this sample is a sync sample (key frame). 
 *                                  For non-video media, the wrapper shall take every sample as sync sample.
 *                                  This flag shall always be SET for this API.
 *
 *                            FLAG_UNCOMPRESSED_SAMPLE
 *                                  Whethter this sample is a uncompressed one. Uncompressed samples shall bypass the decoder.
 *                                  Warning: Video track may have both compressed & uncompressed samples.
 *                                                But some AVI clips seem to abuse this flag, sync samples are mark as uncompressed, 
 *                                                  although they are actually compressed ones.
 *                            
 *                            FLAG_SAMPLE_ERR_CONCEALED
 *                                  There is error in bitstream but a sample is still got by error concealment.                                  
 *                            
 *                            FLAG_SAMPLE_SUGGEST_SEEK
 *                                  A seeking on ALL tracks is suggested although samples can be got by error concealment. 
 *                                  Because there are many corrupts, 
 *                                  and A/V sync is likely impacted by simple concealment(scanning bitstream).
 *
 *                            FLAG_SAMPLE_NOT_FINISHED
 *                                  Sample data output is not finished because the buffer is not big enough.                                 
 *                                  Need to get the remaining data by repetitive calling this func.
 *                                  This feature is only for non-protected clips.
 *
 * @return  PARSER_SUCCESS     An entire sync sample or part of it is got successfully.
 *          PARSER_ERR_NOT_SEEKABLE    No sync sample is got  because the movie is not seekable (index not available) *              
 *          PARSER_EOS      Reaching the end of the track, no sync sample is got.
 *          PARSER_BOS      Reaching the beginning of the track, no sync sample is got.
 *          PARSER_INSUFFICIENT_MEMORY Buffer is too small to hold the sample. 
 *                                  The user can allocate a larger buffer and call this API again (Only for DRM-protected clips).
 *          PARSER_READ_ERROR  File reading error. No need for further error concealment.
 *          PARSER_ERR_CONCEAL_FAIL  There is error in bitstream, and no sample can be found by error concealment. A seeking is helpful.
 *          Others ... Reading fails for other reason.
 */
EXTERN  AVI_PARSER_ERROR_CODE AviGetNextSyncSample(AviParserHandle parserHandle, 
                                        uint32 trackNum,
                                        uint32 direction,
                                        uint8 * sampleData,  
                                        uint32 * dataSize, 
                                        uint64 * usStartTime,
                                        uint64 * usEndTime,
                                        uint32 * flag);



/***************************************************************************************
API Calling Sequence, go ahead only if previous step succeeds:

(0) Check core parser version (optional):
        AviParserVersionInfo()
        
(1) Try to create the parser:
        AviCreateParser()

(2)If the parser is created successfully, parse the file head:
        AviParseHeader()

(3) Check DRM info:
       Whether this movie file is protected by DRM?
        AviIsProtected()

        If protected,
            - Is it a rental file or a purchased file?
            AviQueryContentUsage()

            - Whether protection is required for digital video output signals? 
            AviQueryOutputProtectionFlag()

            - If user want to play protected clip, confirm playback
                AviCommitPlayback()
               Otherwise, delete the parser object and exit.

            
(4) Initialize index table  
    - for the 1st play, intialize the index from the clip, may be time consuming for a large clip
        AviInitializeIndex()  
        
        and export it for fast openning next time (optional)
        AviExportIndex() 

    - for a clip not play for the 1st time and if its index has be exported previously
        AviImportIndex()

        Of course, you call still choose to use AviInitializeIndex()

(5) Now, it's time to get the file, A/V properties, and user data. 
    AviIsSeekable()
    AviGetNumTracks()
    ...
    AviGetMediaType()
    AviGetVideoType()
    AviGetAudioType()
    ...
    AviGetMaxSampleSize()
    AviGetUserData()   
    

    NOTE: AviGetMaxSampleSize() call tell you how big a buffer is enough to hold the largest sample of a track!

(6) Seek to the beginning of the movie.
    Must perform a seeking on All selected tracks, with target time 0 us.
    AviSeek()

    Of course, seek to any time within the play duration is reasonable.

(7) For normal playback (rate = 1X), begin to read A/V samples one by one sequentially from the selected tracks.
    The wrapper decides which track to read
    AviGetNextSample()



(8) For trick mode (FF/RW), can only pick sync sample (video key frames)
    AviGetNextSyncSample() ... for FF
    AviGetPrevSyncSample() ... for RW
 
(9) Reset DRM state when playback ends.Only required for a DRM-protected clip.
    AviFinalizePlayback()


(10) Delete the parser object, as long as AviCreateParser() succeeds
    AviDeleteParser()
 
***************************************************************************************/


#endif //FSL_AVI_PARSER_API_HEADER_INCLUDED

