/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 *
 * Copyright (C) 2007 Andris Pavenis <andris.pavenis@iki.fi>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * ***** END OF LICENSE BLOCK ***** */

#include "mozVoikkoUtils.hxx"
#include "mozVoikkoSpell.hxx"
#include <nsComponentManagerUtils.h>
#include <nsCOMPtr.h>
#include <nsICategoryManager.h>
#include <nsIConsoleService.h>
#include <nsIFile.h>
#include <nsILocalFile.h>
#include <nsIXULRuntime.h>
#include <nsServiceManagerUtils.h>
#include <nsStringAPI.h>
#include <prerror.h>

#if defined(__linux__)

const char *preloadLibNames[] =
{
    "libmalaga.so.7",
    "libvoikko.so.1"
};

const unsigned numPreloadedLibs = sizeof(preloadLibNames)/sizeof(*preloadLibNames);

const char * libvoikkoName = "libvoikko.so.1";

#elif defined(_WIN32)


const char *preloadLibNames[] =
{
    "iconv.dll",
    "intl.dll",
    "libglib-2.0-0.dll",
    "malaga.dll",
    "libvoikko-1.dll"
};

const unsigned numPreloadedLibs = sizeof(preloadLibNames)/sizeof(*preloadLibNames);

const char * libvoikkoName = "libvoikko-1.dll";

#else

#error Target OS not supported

#endif

#define CH(statement, errRetVal) do \
    { \
        nsresult rv = statement; \
        if (NS_FAILED(rv)) \
        { \
            logMessage("%s failed in %s at line %d: error code %X", \
		 #statement, __FILE__, __LINE__, (PRUint32)rv); \
            return errRetVal; \
        } \
    } while(false)


void logMessage(const char *fmt, ...)
{
    va_list args;
    va_start(args, fmt);
    char *msg = PR_vsmprintf(fmt, args);
    va_end(args);

    nsCOMPtr<nsIConsoleService> aConsoleService =
        do_GetService("@mozilla.org/consoleservice;1");

    if (aConsoleService)
    {
        nsCString tmp(msg);
        aConsoleService->LogStringMessage(NS_ConvertUTF8toUTF16(tmp).get());
    }
    else 
    {
        fputs(msg, stdout);
    }
    
    PR_Free(msg);
}

nsCString prGetErrorText(void)
{
    nsCString msg;
    PRInt32 len = PR_GetErrorTextLength();
    if (len > 0)
    {
        char *buffer = (char*)PR_Malloc(len);
        if (buffer)
        {
            PR_GetErrorText(buffer);
            msg = buffer;
            PR_Free(buffer);
        }
    }

    return msg;
}

nsresult checkLeafNameAndGetParent(nsIFile *file, nsIFile **parent,
    const nsACString &expectedLeafName)
{
    nsresult rv;
    nsCString leafName;

    rv = file->GetNativeLeafName(leafName);
    NS_ENSURE_SUCCESS(rv, rv);

    if (leafName.Equals(expectedLeafName))
    {
        rv = file->GetParent(parent);
        NS_ENSURE_SUCCESS(rv, rv);
        return NS_OK;
    }
    else
        return NS_OK;
}

nsresult checkLeafNameAndGetParent(nsIFile *file, nsIFile **parent,
    const nsACString& expectedLeafName1, const nsACString &expectedLeafName2)
{
    nsresult rv;
    nsCString leafName;

    rv = file->GetNativeLeafName(leafName);
    NS_ENSURE_SUCCESS(rv, rv);

    if (leafName.Equals(expectedLeafName1) || leafName.Equals(expectedLeafName2))
    {
        rv = file->GetParent(parent);
        NS_ENSURE_SUCCESS(rv, rv);
        return NS_OK;
    }
    else
        return NS_OK;
}

nsresult findFileInSubdirectory(nsIFile *dir, nsIFile **result,
    const nsACString &subDirName, const nsACString &fileName)
{
    PRBool found;
    nsresult rv;
    nsCOMPtr<nsIFile> tmp;

    *result = nsnull;

    rv = dir->Clone(getter_AddRefs(tmp));
    NS_ENSURE_SUCCESS(rv, rv);

    rv = tmp->AppendNative(subDirName);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = tmp->Exists(&found);
    NS_ENSURE_SUCCESS(rv, rv);
    if (!found)
        return NS_OK;

    rv = tmp->IsDirectory(&found);
    NS_ENSURE_SUCCESS(rv, rv);
    if (!found)
        return NS_OK;

    rv = tmp->AppendNative(fileName);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = tmp->Exists(&found);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = tmp->Clone(result);
    return rv;
}

nsresult getMozVoikkoLibrary(nsIFile **file)
{
    nsresult rv;
    char *mozVoikkoLibPathC;
  
    *file = nsnull;

    nsCOMPtr<nsICategoryManager> catMgr =
        do_GetService(NS_CATEGORYMANAGER_CONTRACTID);

    if (!catMgr)
    {
        logMessage("Failed to get nsICategoryManager");
        return NS_ERROR_FAILURE;
    }

    rv = catMgr->GetCategoryEntry("spell-check-engine", MOZ_VOIKKOSPELL_CONTRACTID,
        &mozVoikkoLibPathC);
    if (NS_FAILED(rv)) 
    {
        logMessage("Failed to get category entry for mozVoikko");
        return rv;
    }

    nsCString path(mozVoikkoLibPathC);

    nsCOMPtr<nsILocalFile> mozVoikkoLib;
    mozVoikkoLib = do_CreateInstance(NS_LOCAL_FILE_CONTRACTID, &rv);
    if (!mozVoikkoLib)
    {
        logMessage("Failed to create instance of nsILoclFile");
        return NS_ERROR_FAILURE;
    }

    rv = mozVoikkoLib->InitWithPath(NS_ConvertUTF8toUTF16(path));
    if (NS_FAILED(rv))
    {
        logMessage("Failed to set path for nsILocalFile");
        return rv;
    }

    rv = mozVoikkoLib->Clone(file);
    if (NS_FAILED(rv))
    { 
        logMessage("%s: Failed to clone nsIFile", __FUNCTION__);
        return rv;
    }
    
    return NS_OK;
}

nsresult getMozVoikkoBaseDirs(nsIFile **libDir, nsIFile **dataDir)
{
    nsresult rv;
    nsCOMPtr<nsIFile> mozVoikkoLibName;
    nsCOMPtr<nsIFile> currDir, parentDir;
    nsCOMPtr<nsIFile> tmpDir, tmpFile;
    nsCString leafName, tmp;
    nsCString shortTargetName;
    nsCString fullTargetName;

    // Get platform name (eg. WINNT_x86-msvc or Linux_x86_64-gcc3).
    // WARNING: not frozen interface nsIXULRuntime is being used.
    // String shortTargetName will contain only target OS name (like WINNT)
    // String fullTargetName will contain additionally XPCOMAPI_NAME

    nsCOMPtr<nsIXULRuntime> runtime = do_GetService("@mozilla.org/xre/app-info;1", &rv);

    NS_ENSURE_SUCCESS(rv, rv);

    rv = runtime->GetOS(shortTargetName);
    NS_ENSURE_SUCCESS(rv, rv);

    fullTargetName = shortTargetName;

    rv = runtime->GetXPCOMABI(tmp);
    if (NS_SUCCEEDED(rv))
    {
        fullTargetName.Append("_");
        fullTargetName.Append(tmp);
    }
 
    // Get path of mozvoikko shared library.
    // mozVoikkoSpell::registerExtension() has saved it using nsICategory manager
    // when registrating the extension
    rv = getMozVoikkoLibrary(getter_AddRefs(mozVoikkoLibName));
    NS_ENSURE_SUCCESS(rv, rv);

    // Extract directory part
    rv = mozVoikkoLibName->GetParent(getter_AddRefs(currDir));
    NS_ENSURE_SUCCESS(rv, rv);

    rv = checkLeafNameAndGetParent(currDir, getter_AddRefs(parentDir),
        NS_LITERAL_CSTRING("components"));
    NS_ENSURE_SUCCESS(rv, rv);
    if (!parentDir)
        return NS_ERROR_FAILURE;

    currDir = parentDir;

    rv = findFileInSubdirectory(currDir, getter_AddRefs(tmpFile),
        NS_LITERAL_CSTRING("voikko"), NS_LITERAL_CSTRING("voikko-fi_FI.pro"));
    NS_ENSURE_SUCCESS(rv, rv);
    
    if (!tmpFile)
    {
        rv = checkLeafNameAndGetParent(currDir, getter_AddRefs(parentDir),
            shortTargetName, fullTargetName);
        NS_ENSURE_SUCCESS(rv, rv);
        if (parentDir)
        {
            currDir = parentDir;

            rv = checkLeafNameAndGetParent(currDir, getter_AddRefs(parentDir),
                NS_LITERAL_CSTRING("platform"));
            NS_ENSURE_SUCCESS(rv, rv);

            if (parentDir)
            {
                rv = findFileInSubdirectory(parentDir, getter_AddRefs(tmpFile),
                    NS_LITERAL_CSTRING("voikko"),
                    NS_LITERAL_CSTRING("voikko-fi_FI.pro"));
                NS_ENSURE_SUCCESS(rv, rv);
            }
        }
    }

    if (!tmpFile)
    {
        return NS_ERROR_FAILURE;
    }

    rv = tmpFile->GetParent(dataDir);
    NS_ENSURE_SUCCESS(rv, rv);

    rv = (*dataDir)->Clone(getter_AddRefs(tmpDir));
    NS_ENSURE_SUCCESS(rv, rv);

    rv = findFileInSubdirectory(tmpDir, getter_AddRefs(tmpFile), fullTargetName,
        nsCString(libvoikkoName));
    NS_ENSURE_SUCCESS(rv, rv);

    if (tmpFile)
    {
        rv = tmpFile->GetParent(libDir);
        return rv;
    }

    rv = findFileInSubdirectory(tmpDir, getter_AddRefs(tmpFile), shortTargetName,
        nsCString(libvoikkoName));
    NS_ENSURE_SUCCESS(rv, rv);

    if (tmpFile)
    {
        rv = tmpFile->GetParent(libDir);
        return rv;
    }

    return NS_ERROR_FAILURE;
}

nsresult loadExtraLibraries(int numLibs, const char **names)
{
    nsresult rv;
    nsCOMPtr<nsIFile> base, baseDir;
    nsCString tmp;

    rv = getMozVoikkoLibrary(getter_AddRefs(base));
    if (NS_FAILED(rv))
        return rv; 

    base->GetParent(getter_AddRefs(baseDir));

    base->GetNativePath(tmp);
    logMessage("mozVoikko.so path is %s", tmp.get());

    baseDir->GetNativePath(tmp);
    logMessage("mozVoikko.so directory is %s", tmp.get());

    return NS_OK;
}

PreloadedLibraries::PreloadedLibraries(nsIFile *libDir, const char **names, unsigned numLibs) :
    ok(false), numLibs(numLibs), libraries(NULL)
{
    nsresult rv;

    libraries = new PRLibrary *[numLibs];
    if (!libraries)
        return;

    memset(libraries, 0, sizeof(PRLibrary*)*numLibs);

    for (unsigned i = 0; i < numLibs; i++)
    {
        PRBool exists;
        nsCOMPtr<nsIFile> libFile;
        nsCString libPath;

        rv = libDir->Clone(getter_AddRefs(libFile));
        if (NS_FAILED(rv))
            return;

        rv = libFile->AppendNative(nsCString(names[i]));
        if (NS_FAILED(rv))
            return;

        rv = libFile->GetNativePath(libPath);
        if (NS_FAILED(rv))
            return;

        rv = libFile->Exists(&exists);
        if (NS_FAILED(rv))
            return;

        if (exists)
        {
            libraries[i] = PR_LoadLibrary(libPath.get());
            if (!libraries[i])
            {
                logMessage("mozVoikko: failed to load library %s: %s",
                           libPath.get(), prGetErrorText().get());
                return;
            }
        }
        else
        {
            libraries[i] = nsnull;
        }

        logMessage("Loaded %s", libPath.get()); 
    }

    ok = true;
}

PreloadedLibraries::~PreloadedLibraries()
{
    if (libraries)
    {
        for (unsigned i = numLibs; i > 0; i--)
        {
            if (libraries[i-1])
                PR_UnloadLibrary(libraries[i-1]);
        }

        delete [] libraries;
    }
}
