/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gdk-pixbuf/gdk-pixbuf.h>
#include "pgmglimage.h"

GST_DEBUG_CATEGORY_STATIC (pgm_gl_image_debug);
#define GST_CAT_DEFAULT pgm_gl_image_debug

static PgmGlDrawableClass *parent_class = NULL;

/* Private methods */

/* Computes the image pixel aspect ratio */
static void
update_image_ratio (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);

  GST_OBJECT_LOCK (image);
  glimage->image_ratio = (gfloat) image->par_n / image->par_d;
  GST_OBJECT_UNLOCK (image);
}

/* Computes the drawable pixel aspect ratio */
static void
update_drawable_ratio (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);

  if (G_LIKELY (gldrawable->height != 0.0))
    glimage->drawable_ratio = gldrawable->width / gldrawable->height;
  else
    glimage->drawable_ratio = 1.0f;
}

/* Updates the last position fields with the one from the drawable */
static void
update_last_position (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);

  glimage->last_x = gldrawable->x;
  glimage->last_y = gldrawable->y;
  glimage->last_z = gldrawable->z;
}

/* Updates the alignment */
static void
update_alignment (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmImageAlignment align;

  GST_OBJECT_LOCK (image);
  align = image->align;
  GST_OBJECT_UNLOCK (image);

  /* Horizontal alignment */
  if (align & PGM_IMAGE_LEFT)
    glimage->h_align = 0.0f;
  else if (align & PGM_IMAGE_RIGHT)
    glimage->h_align = 1.0f;
  else
    glimage->h_align = 0.5f;

  /* Vertical alignment */
  if (align & PGM_IMAGE_TOP)
    glimage->v_align = 0.0f;
  else if (align & PGM_IMAGE_BOTTOM)
    glimage->v_align = 1.0f;
  else
    glimage->v_align = 0.5f;
}

/* Updates the interpolation */
static void
update_interp (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmTexture *texture = glimage->texture;

  GST_OBJECT_LOCK (image);

  if (image->interp == PGM_IMAGE_BILINEAR)
    texture->filter = PGM_GL_LINEAR;
  else if (image->interp == PGM_IMAGE_NEAREST)
    texture->filter = PGM_GL_NEAREST;

  GST_OBJECT_UNLOCK (image);
}

/* Updates the wrapping */
static void
update_wrapping (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmTexture *texture = glimage->texture;

  GST_OBJECT_LOCK (image);

  if (image->wrap_s == PGM_IMAGE_CLAMP)
    texture->wrap_s = PGM_GL_CLAMP_TO_EDGE;
  else if (image->wrap_s == PGM_IMAGE_REPEAT)
    texture->wrap_s = PGM_GL_REPEAT;
  else if (image->wrap_s == PGM_IMAGE_TRANSPARENT)
    texture->wrap_s = PGM_GL_CLAMP_TO_BORDER;

  if (image->wrap_t == PGM_IMAGE_CLAMP)
    texture->wrap_t = PGM_GL_CLAMP_TO_EDGE;
  else if (image->wrap_t == PGM_IMAGE_REPEAT)
    texture->wrap_t = PGM_GL_REPEAT;
  else if (image->wrap_t == PGM_IMAGE_TRANSPARENT)
    texture->wrap_t = PGM_GL_CLAMP_TO_BORDER;

  GST_OBJECT_UNLOCK (image);
}

/* Updates the mapping (texture) matrix */
static void
update_mapping_matrix (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);

  GST_OBJECT_LOCK (image);
  pgm_texture_set_matrix (glimage->texture, image->mapping_matrix);
  GST_OBJECT_UNLOCK (image);
}

/* Updates the color of the inner border vertices */
static void
update_border_inner_color (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmGlFloat *border_color = glimage->border_color;
  PgmImage *image = PGM_IMAGE (drawable);
  gfloat r, g, b, a;
  guint i;

  /* Get the right color components */
  GST_OBJECT_LOCK (drawable);
  r = image->border_inner_r * INV_255;
  g = image->border_inner_g * INV_255;
  b = image->border_inner_b * INV_255;
  a = image->border_inner_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  /* Then set the color attribute of the inner vertices */
  for (i = 0; i < 40; i += 8)
    {
      border_color[i+0] = r;
      border_color[i+1] = g;
      border_color[i+2] = b;
      border_color[i+3] = a;
    }
}

/* Updates the color of the outer border vertices */
static void
update_border_outer_color (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmGlFloat *border_color = glimage->border_color;
  PgmImage *image = PGM_IMAGE (drawable);
  gfloat r, g, b, a;
  guint i;

  /* Get the right color components */
  GST_OBJECT_LOCK (drawable);
  r = image->border_outer_r * INV_255;
  g = image->border_outer_g * INV_255;
  b = image->border_outer_b * INV_255;
  a = image->border_outer_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  /* Then set the color attribute of the outer vertices */
  for (i = 4; i < 44; i += 8)
    {
      border_color[i+0] = r;
      border_color[i+1] = g;
      border_color[i+2] = b;
      border_color[i+3] = a;
    }
}

/* Updates the border vertices */
static void
set_border_vertices (PgmGlImage *glimage,
                     gfloat border_width,
                     gfloat border_height)
{
  PgmGlFloat *border_vertex = glimage->border_vertex;
  PgmGlFloat *img_vertex = glimage->vertex;

  /* Here is how the border vertices are drawn using a GL_TRIANGLE_STRIP mode.
   * Note that the vertices 8 and 9 are not shown but are using the exact same
   * coordinates than respectively the vertices 0 and 1. Remember that the
   * projection matrix flip the scene vertically and the representation of this
   * schema in a Pigment scene is inverted along the y cardinal axis.
   *
   * 7------------5
   * | 6--------4 |
   * | |        | |
   * | 0--------2 |
   * 1------------3
   */

  /* Vertex 0 */
  border_vertex[0] = img_vertex[0];
  border_vertex[1] = img_vertex[1];
  border_vertex[2] = img_vertex[2];
  /* Vertex 1 */
  border_vertex[3] = img_vertex[0] - border_width;
  border_vertex[4] = img_vertex[1] - border_height;
  border_vertex[5] = img_vertex[2];
  /* Vertex 2 */
  border_vertex[6] = img_vertex[3];
  border_vertex[7] = img_vertex[4];
  border_vertex[8] = img_vertex[5];
  /* Vertex 3 */
  border_vertex[9] = img_vertex[3] + border_width;
  border_vertex[10] = img_vertex[4] - border_height;
  border_vertex[11] = img_vertex[5];
  /* Vertex 4 */
  border_vertex[12] = img_vertex[6];
  border_vertex[13] = img_vertex[7];
  border_vertex[14] = img_vertex[8];
  /* Vertex 5 */
  border_vertex[15] = img_vertex[6] + border_width;
  border_vertex[16] = img_vertex[7] + border_height;
  border_vertex[17] = img_vertex[8];
  /* Vertex 6 */
  border_vertex[18] = img_vertex[9];
  border_vertex[19] = img_vertex[10];
  border_vertex[20] = img_vertex[11];
  /* Vertex 7 */
  border_vertex[21] = img_vertex[9] - border_width;
  border_vertex[22] = img_vertex[10] + border_height;
  border_vertex[23] = img_vertex[11];
  /* Vertex 8, simple copy of the vertex 0 */
  border_vertex[24] = border_vertex[0];
  border_vertex[25] = border_vertex[1];
  border_vertex[26] = border_vertex[2];
  /* Vertex 9, simple copy of the vertex 1 */
  border_vertex[27] = border_vertex[3];
  border_vertex[28] = border_vertex[4];
  border_vertex[29] = border_vertex[5];
}

/* Simply copies gldrawable background vertices */
static void
set_image_standard_vertices (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlFloat *drb_vertex = gldrawable->bg_vertex;
  PgmGlFloat *img_vertex = glimage->vertex;
  gfloat border_width, border_height;

  /* Get the border sizes avoiding overlapped borders */
  border_width = MIN (glimage->border_width, gldrawable->width / 2.0f);
  border_height = MIN (glimage->border_width, gldrawable->height / 2.0f);

  /* Copy the drawable vertices applying the border offset */
  img_vertex[0] = drb_vertex[0] + border_width;
  img_vertex[1] = drb_vertex[1] + border_height;
  img_vertex[2] = drb_vertex[2];
  img_vertex[3] = drb_vertex[3] - border_width;
  img_vertex[4] = img_vertex[1];
  img_vertex[5] = drb_vertex[5];
  img_vertex[6] = img_vertex[3];
  img_vertex[7] = drb_vertex[7] - border_height;
  img_vertex[8] = drb_vertex[8];
  img_vertex[9] = img_vertex[0];
  img_vertex[10] = img_vertex[7];
  img_vertex[11] = drb_vertex[11];

  /* Update the border vertices if necessary */
  if (glimage->border_width > 0.0f)
    set_border_vertices (glimage, border_width, border_height);
}

/* Modifies vertices for the scaled layout */
static void
set_image_scaled_vertices (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlFloat *drb_vertex = gldrawable->bg_vertex;
  PgmGlFloat *img_vertex = glimage->vertex;
  gfloat border_width, border_height, offset;

  /* Drawable width larger than image width */
  if (glimage->drawable_ratio >= glimage->image_ratio)
    {
      offset = gldrawable->width - gldrawable->height * glimage->image_ratio;

      /* Get the border sizes avoiding overlapped borders */
      border_width = MIN (glimage->border_width,
                          (gldrawable->width - offset) / 2.0f);
      border_height = MIN (glimage->border_width, gldrawable->height / 2.0f);

      img_vertex[0] = (drb_vertex[0] + offset *
                       glimage->h_align) + border_width;
      img_vertex[1] = drb_vertex[1] + border_height;
      img_vertex[3] = (drb_vertex[3] - offset *
                       (1.0f - glimage->h_align)) - border_width;
      img_vertex[4] = drb_vertex[4] + border_height;
      img_vertex[6] = img_vertex[3];
      img_vertex[7] = drb_vertex[7] - border_height;
      img_vertex[9] = img_vertex[0];
      img_vertex[10] = drb_vertex[10] - border_height;
    }
  /* Drawable height larger than image height */
  else
    {
      offset = gldrawable->height - gldrawable->width / glimage->image_ratio;

      /* Get the border sizes avoiding overlapped borders */
      border_width = MIN (glimage->border_width, gldrawable->width / 2.0f);
      border_height = MIN (glimage->border_width,
                           (gldrawable->height - offset) / 2.0f);

      img_vertex[0] = drb_vertex[0] + border_width;
      img_vertex[1] = (drb_vertex[1] + offset *
                       glimage->v_align) + border_height;
      img_vertex[3] = drb_vertex[3] - border_width;
      img_vertex[4] = img_vertex[1];
      img_vertex[6] = drb_vertex[6] - border_width;
      img_vertex[7] = (drb_vertex[7] - offset *
                       (1.0f - glimage->v_align)) - border_height;
      img_vertex[9] = drb_vertex[9] + border_width;
      img_vertex[10] = img_vertex[7];
    }

  /* Then simply copy the z position from the background one */
  img_vertex[2] = drb_vertex[2];
  img_vertex[5] = drb_vertex[5];
  img_vertex[8] = drb_vertex[8];
  img_vertex[11] = drb_vertex[11];

  /* Update the border vertices if necessary */
  if (glimage->border_width > 0.0f)
    set_border_vertices (glimage, border_width, border_height);
}

/* Modifies texture coordinates for the zoomed layout */
static void
set_image_zoomed_coordinates (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmTexture *texture = glimage->texture;
  gfloat max_s, max_t;
  gfloat offset;

  max_s = (gfloat) texture->width / texture->width_pot;
  max_t = (gfloat) texture->height / texture->height_pot;

  /* Adapt coordinates when drawable width is larger than image width */
  if (glimage->drawable_ratio >= glimage->image_ratio)
    {
      gfloat image_height = gldrawable->width / glimage->image_ratio;

      offset = ((image_height - gldrawable->height) * max_t) / image_height;
      glimage->coord[0] = 0.0f;
      glimage->coord[1] = offset * glimage->v_align;
      glimage->coord[2] = max_s;
      glimage->coord[3] = glimage->coord[1];
      glimage->coord[4] = max_s;
      glimage->coord[5] = max_t - offset * (1.0f - glimage->v_align);
      glimage->coord[6] = 0.0f;
      glimage->coord[7] = glimage->coord[5];
    }
  /* Adapt coordinates when drawable height is larger than image height */
  else
    {
      gfloat image_width = gldrawable->height * glimage->image_ratio;

      offset = ((image_width - gldrawable->width) * max_s) / image_width;
      glimage->coord[0] = offset * glimage->h_align;
      glimage->coord[1] = 0.0f;
      glimage->coord[2] = max_s - offset * (1.0f - glimage->h_align);
      glimage->coord[3] = 0.0f;
      glimage->coord[4] = glimage->coord[2];
      glimage->coord[5] = max_t;
      glimage->coord[6] = glimage->coord[0];
      glimage->coord[7] = max_t;
    }
}

/* Modifies texture coordinates to fill the whole stored texture */
static void
set_image_standard_coordinates (PgmGlImage *glimage)
{
  PgmTexture *texture = glimage->texture;

  glimage->coord[0] = 0.0f;
  glimage->coord[1] = 0.0f;
  glimage->coord[2] = (gfloat) texture->width / texture->width_pot;
  glimage->coord[3] = 0.0f;
  glimage->coord[4] = glimage->coord[2];
  glimage->coord[5] = (gfloat) texture->height / texture->height_pot;
  glimage->coord[6] = 0.0f;
  glimage->coord[7] = glimage->coord[5];
}

/* Adapts properties to a size change depending on the layout */
static void
update_layout (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmImageLayoutType layout;

  GST_OBJECT_LOCK (image);
  layout = image->layout;
  GST_OBJECT_UNLOCK (image);

  switch (layout)
    {
    case PGM_IMAGE_SCALED:
    case PGM_IMAGE_CENTERED: /* FIXME: Not implemented yet */
    case PGM_IMAGE_TILED:    /* FIXME: Not implemented yet */
      set_image_scaled_vertices (glimage);
      set_image_standard_coordinates (glimage);
      break;

    case PGM_IMAGE_ZOOMED:
      set_image_standard_vertices (glimage);
      set_image_zoomed_coordinates (glimage);
      break;

    case PGM_IMAGE_FILLED:
      set_image_standard_vertices (glimage);
      set_image_standard_coordinates (glimage);
      break;

    default:
      break;
    }
}

/* Adapts to a drawable position change */
static void
update_vertices_position (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlFloat *img_vertex = glimage->vertex;
  PgmGlFloat *img_border = glimage->border_vertex;
  gfloat x_offset, y_offset, z_offset;
  guint i;

  x_offset = gldrawable->x - glimage->last_x;
  y_offset = gldrawable->y - glimage->last_y;
  z_offset = gldrawable->z - glimage->last_z;

  /* Add the offsets to the current vertices */
  for (i = 0; i < 12; i += 3)
    {
      img_vertex[i] += x_offset;
      img_vertex[i+1] += y_offset;
      img_vertex[i+2] += z_offset;
    }

  /* Add the offsets to the border vertices, if necessary */
  if (glimage->border_width > 0.0f)
    for (i = 0; i < 30; i += 3)
      {
        img_border[i] += x_offset;
        img_border[i+1] += y_offset;
        img_border[i+2] += z_offset;
      }

  /* And update the last position with the new one */
  update_last_position (glimage);
}

/* Update the border_width value stored in the GL image */
static void
update_border_width (PgmGlImage *glimage)
{
  PgmDrawable *drawable = PGM_GL_DRAWABLE (glimage)->drawable;
  PgmImage *image = PGM_IMAGE (drawable);

  GST_OBJECT_LOCK (image);
  glimage->border_width = image->border_width;
  GST_OBJECT_UNLOCK (image);
}

/* Update the image ratio and layout of all the slaves of the given image */
static void
update_slaves (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmImage *master = PGM_IMAGE (gldrawable->drawable);
  PgmGlImage *slave;
  GList *walk;

  GST_OBJECT_LOCK (master);

  walk = master->slaves;
  while (walk)
    {
      /* Retrieve the slave glimage */
      GST_OBJECT_LOCK (glviewport);
      slave = g_hash_table_lookup (glviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glviewport);

      /* Update slave glimage parameters */
      if (slave)
        {
          GST_OBJECT_UNLOCK (master);
          update_image_ratio (slave);
          update_layout (slave);
          GST_OBJECT_LOCK (master);
        }

      walk = walk->next;
    }

  GST_OBJECT_UNLOCK (master);
}

/* PgmGlDrawable methods */

static void
pgm_gl_image_draw (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmContextProcAddress *gl;

  GST_LOG_OBJECT (gldrawable, "draw");

  /* Avoid drawing if it's not visible */
  if (glimage->empty || (glimage->fg_color[3] == 0.0f))
    return;

  gl = gldrawable->glviewport->context->gl;

  gl->enable_client_state (PGM_GL_VERTEX_ARRAY);
  gl->enable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);

  /* Upload vertices and texture coordinates for the image */
  gl->vertex_pointer (3, PGM_GL_FLOAT, 0, glimage->vertex);
  gl->tex_coord_pointer (2, PGM_GL_FLOAT, 0, glimage->coord);
  gl->color_4fv (glimage->fg_color);

  /* Draw the image */
  pgm_texture_bind (glimage->texture);
  gl->draw_arrays (PGM_GL_QUADS, 0, 4);
  pgm_texture_unbind (glimage->texture);

  gl->disable_client_state (PGM_GL_TEXTURE_COORD_ARRAY);

  if (glimage->border_width > 0.0f)
    {
      /* Upload vertices and colors for the border */
      gl->enable_client_state (PGM_GL_COLOR_ARRAY);
      gl->vertex_pointer (3, PGM_GL_FLOAT, 0, glimage->border_vertex);
      gl->color_pointer (4, PGM_GL_FLOAT, 0, glimage->border_color);

      /* Draw the border */
      gl->draw_arrays (PGM_GL_QUAD_STRIP, 0, 10);
      gl->disable_client_state (PGM_GL_COLOR_ARRAY);
    }

  gl->disable_client_state (PGM_GL_VERTEX_ARRAY);
}

static void
pgm_gl_image_regenerate (PgmGlDrawable *gldrawable)
{
  GST_LOG_OBJECT (gldrawable, "regenerate");
}

static void
pgm_gl_image_update_projection (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);

  GST_LOG_OBJECT (gldrawable, "update_projection");

  update_image_ratio (glimage);
  update_layout (glimage);
}

static void
pgm_gl_image_set_size (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "set_size");

  update_drawable_ratio (glimage);
  update_layout (glimage);
}

static void
pgm_gl_image_set_position (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);

  GST_DEBUG_OBJECT (gldrawable, "set_position");

  update_vertices_position (glimage);
}

static void
pgm_gl_image_set_fg_color (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmDrawable *drawable;

  GST_DEBUG_OBJECT (gldrawable, "set_fg_color");

  drawable = gldrawable->drawable;

  GST_OBJECT_LOCK (drawable);
  glimage->fg_color[0] = drawable->fg_r * INV_255;
  glimage->fg_color[1] = drawable->fg_g * INV_255;
  glimage->fg_color[2] = drawable->fg_b * INV_255;
  glimage->fg_color[3] = drawable->fg_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);
}

static void
pgm_gl_image_set_opacity (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmDrawable *drawable = gldrawable->drawable;
  PgmImage *image = PGM_IMAGE (drawable);
  PgmGlFloat *border_color = glimage->border_color;
  gfloat fg_opacity, border_opacity;
  guint i;

  GST_DEBUG_OBJECT (gldrawable, "set_opacity");

  /* Get the opacities */
  GST_OBJECT_LOCK (drawable);
  fg_opacity = drawable->fg_a * drawable->opacity * SQR_INV_255;
  border_opacity = image->border_inner_a * drawable->opacity * SQR_INV_255;
  GST_OBJECT_UNLOCK (drawable);

  /* Image alpha blending */
  glimage->fg_color[3] = fg_opacity;

  /* Border alpha blending */
  for (i = 3; i < 40; i += 4)
    border_color[i] = border_opacity;
}

static void
pgm_gl_image_sync (PgmGlDrawable *gldrawable)
{
  PgmGlImage *glimage = PGM_GL_IMAGE (gldrawable);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmImageStorageType type;

  GST_LOG_OBJECT (gldrawable, "sync");

  /* Synchronize various properties */
  pgm_gl_image_set_fg_color (gldrawable);
  update_interp (glimage);
  update_wrapping (glimage);
  update_alignment (glimage);
  update_last_position (glimage);
  update_drawable_ratio (glimage);
  update_image_ratio (glimage);
  update_mapping_matrix (glimage);
  update_border_width (glimage);
  update_border_inner_color (glimage);
  update_border_outer_color (glimage);

  GST_OBJECT_LOCK (image);
  type = image->storage_type;
  GST_OBJECT_UNLOCK (image);

  /* Synchronize the buffer */
  switch (type)
    {
    case PGM_IMAGE_FILE:
      pgm_gl_image_set_from_file (glimage);
      break;
    case PGM_IMAGE_BUFFER:
      pgm_gl_image_set_from_buffer (glimage);
      break;
    case PGM_IMAGE_GST_BUFFER:
      pgm_gl_image_set_from_gst_buffer (glimage);
      break;
    case PGM_IMAGE_PIXBUF:
      pgm_gl_image_set_from_pixbuf (glimage);
      break;
    case PGM_IMAGE_SYSTEM_BUFFER:
      pgm_gl_image_set_from_system_buffer (glimage);
      break;
    case PGM_IMAGE_IMAGE:
      pgm_gl_image_set_from_image (glimage);
      break;
    case PGM_IMAGE_EMPTY:
      break;
    default:
      break;
    }
}

/* GObject stuff */

PGM_DEFINE_DYNAMIC_TYPE (PgmGlImage, pgm_gl_image, PGM_TYPE_GL_DRAWABLE);

void
pgm_gl_image_register (GTypeModule *module)
{
  pgm_gl_image_register_type (module);
}

static void
pgm_gl_image_dispose (GObject *object)
{

  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (object);
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmGlImage *glimage = PGM_GL_IMAGE (object);
  PgmImage *master = PGM_IMAGE (gldrawable->drawable);
  PgmGlImage *slave;
  GList *walk;

  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "dispose");

  /* Clean up the slaves */
  GST_OBJECT_LOCK (master);
  walk = master->slaves;
  while (walk)
    {
      GST_OBJECT_LOCK (glviewport);
      slave = g_hash_table_lookup (glviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glviewport);

      if (slave)
        {
          slave->empty = TRUE;
          slave->texture = slave->native_texture;
          slave->image_ratio = 0.0f;
        }

      walk = walk->next;
    }
  GST_OBJECT_UNLOCK (master);

  gst_object_unref (gldrawable->drawable);
  pgm_context_remove_tasks_with_data (gldrawable->glviewport->context,
                                      glimage->native_texture);
  task = pgm_context_task_new (PGM_CONTEXT_FREE_TEXTURE,
                               glimage->native_texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  glimage->native_texture = NULL;
  glimage->texture = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_gl_image_class_init (PgmGlImageClass *klass)
{
  GObjectClass *gobject_class;
  PgmGlDrawableClass *drawable_class;

  GST_DEBUG_CATEGORY_INIT (pgm_gl_image_debug, "pgm_gl_image", 0,
                           "OpenGL plugin: PgmGlImage");

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  drawable_class = PGM_GL_DRAWABLE_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_gl_image_dispose);

  /* PgmGlDrawable virtual table */
  drawable_class->sync = GST_DEBUG_FUNCPTR (pgm_gl_image_sync);
  drawable_class->draw = GST_DEBUG_FUNCPTR (pgm_gl_image_draw);
  drawable_class->regenerate = GST_DEBUG_FUNCPTR (pgm_gl_image_regenerate);
  drawable_class->update_projection =
    GST_DEBUG_FUNCPTR (pgm_gl_image_update_projection);
  drawable_class->set_size = GST_DEBUG_FUNCPTR (pgm_gl_image_set_size);
  drawable_class->set_position = GST_DEBUG_FUNCPTR (pgm_gl_image_set_position);
  drawable_class->set_fg_color = GST_DEBUG_FUNCPTR (pgm_gl_image_set_fg_color);
  drawable_class->set_opacity = GST_DEBUG_FUNCPTR (pgm_gl_image_set_opacity);
}

static void
pgm_gl_image_class_finalize (PgmGlImageClass *klass)
{
  return;
}

static void
pgm_gl_image_init (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "init");

  glimage->empty = TRUE;
  glimage->border_width = 0.0f;
}

/* Public methods */

PgmGlDrawable *
pgm_gl_image_new (PgmDrawable *drawable,
                  PgmGlViewport *glviewport)
{
  PgmImage *master = PGM_IMAGE (drawable);
  PgmGlImage *slave;
  PgmGlImage *glimage;
  PgmGlDrawable *gldrawable;
  GList *walk;

  glimage = g_object_new (PGM_TYPE_GL_IMAGE, NULL);

  GST_DEBUG_OBJECT (glimage, "created new glimage");

  glimage->native_texture = pgm_texture_new (glviewport->context);
  glimage->texture = glimage->native_texture;

  gldrawable = PGM_GL_DRAWABLE (glimage);
  gldrawable->drawable = gst_object_ref (drawable);
  gldrawable->glviewport = glviewport;
  pgm_gl_viewport_connect_changed_callback (glviewport, gldrawable);
  pgm_gl_drawable_sync (gldrawable);

  /* Update the slaves that are in the Canvas */
  GST_OBJECT_LOCK (master);
  walk = master->slaves;
  while (walk)
    {
      GST_OBJECT_LOCK (glviewport);
      slave = g_hash_table_lookup (glviewport->drawable_hash, walk->data);
      GST_OBJECT_UNLOCK (glviewport);

      if (slave)
        {
          slave->empty = FALSE;
          slave->texture = glimage->texture;
          update_image_ratio (slave);
          update_layout (slave);
        }

      walk = walk->next;
    }
  GST_OBJECT_UNLOCK (master);

  return gldrawable;
}

void
pgm_gl_image_clear (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmContextTask *task = NULL;

  GST_DEBUG_OBJECT (glimage, "clear");

  if (glimage->empty)
    return;

  /* Request texture clean up in case it was not using a master texture */
  if (glimage->texture == glimage->native_texture)
    {
      task = pgm_context_task_new (PGM_CONTEXT_CLEAN_TEXTURE, glimage->texture);
      pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
    }
  else
    glimage->texture = glimage->native_texture;

  GST_OBJECT_LOCK (glimage);

  glimage->empty = TRUE;
  glimage->image_ratio = 0.0f;

  GST_OBJECT_UNLOCK (glimage);
}

void
pgm_gl_image_set_from_file (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_file");

  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_FILE
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  if (image->data.file.pixbuf == NULL)
    {
      GST_OBJECT_UNLOCK (image);
      _pgm_image_stored_from_file_load (image);
      return;
    }

  pgm_texture_set_pixbuf (glimage->texture, image->data.file.pixbuf);
  pgm_texture_set_matrix (glimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  _pgm_image_stored_from_file_free (image);

  glimage->empty = FALSE;
  update_image_ratio (glimage);
  update_layout (glimage);
  update_slaves (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_buffer (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_buffer");

  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_BUFFER
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  pgm_texture_set_buffer (glimage->texture, image->data.buffer.buffer,
                          image->data.buffer.format, image->data.buffer.width,
                          image->data.buffer.height, image->data.buffer.size,
                          image->data.buffer.stride);
  pgm_texture_set_matrix (glimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  glimage->empty = FALSE;

  update_image_ratio (glimage);
  update_layout (glimage);
  update_slaves (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_gst_buffer (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_gst_buffer");

  /* It's the first time the glimage receives a GstBuffer, let's set the
   * texture buffer with its different informations */
  if (G_UNLIKELY (glimage->empty))
    {
      GST_OBJECT_LOCK (image);

      if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_GST_BUFFER
                        || image->storage_type == PGM_IMAGE_IMAGE)))
        {
          GST_OBJECT_UNLOCK (image);
          return;
        }

      pgm_texture_set_gst_buffer (glimage->texture,
                                  image->data.gst_buffer.gst_buffer,
                                  image->data.gst_buffer.format,
                                  image->data.gst_buffer.width,
                                  image->data.gst_buffer.height,
                                  image->data.gst_buffer.stride);
      pgm_texture_set_matrix (glimage->texture, image->mapping_matrix);

      GST_OBJECT_UNLOCK (image);

      glimage->empty = FALSE;
      update_image_ratio (glimage);
      update_layout (glimage);
      update_slaves (glimage);

      task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
      pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
    }

  /* It's not the first time, so we just need to send the updated buffer */
  else
    {
      GST_OBJECT_LOCK (image);

      if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_GST_BUFFER
                        || image->storage_type == PGM_IMAGE_IMAGE)))
        {
          GST_OBJECT_UNLOCK (image);
          return;
        }

      pgm_texture_update_gst_buffer (glimage->texture,
                                     image->data.gst_buffer.gst_buffer);
      GST_OBJECT_UNLOCK (image);
    }

  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_pixbuf (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_pixbuf");

  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_PIXBUF
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  pgm_texture_set_pixbuf (glimage->texture, image->data.pixbuf.pixbuf);
  pgm_texture_set_matrix (glimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  glimage->empty = FALSE;
  update_image_ratio (glimage);
  update_layout (glimage);
  update_slaves (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
  task = pgm_context_task_new (PGM_CONTEXT_UPLOAD_TEXTURE, glimage->texture);
  pgm_context_push_deferred_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_system_buffer (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_from_system_buffer");

  GST_OBJECT_LOCK (image);

  if (!(gldrawable->glviewport->context->feature_mask
        & PGM_GL_FEAT_TEXTURE_NON_POWER_OF_TWO))
    {
      GST_DEBUG_OBJECT (image, "System buffer cannot be set, the OpenGL "
                        "implementation does not support NPOT textures");
      GST_OBJECT_UNLOCK (image);
      return;
    }

  if (!(image->data.system_buffer.format == PGM_IMAGE_RGB
        || image->data.system_buffer.format == PGM_IMAGE_RGBA))
    {
      GST_DEBUG_OBJECT (image, "System buffer cannot be set, the OpenGL plugin "
                        "only supports RGB and RGBA color spaces");
      GST_OBJECT_UNLOCK (image);
      return;
    }

  if (G_UNLIKELY (!(image->storage_type == PGM_IMAGE_SYSTEM_BUFFER
                    || image->storage_type == PGM_IMAGE_IMAGE)))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  pgm_texture_set_system_buffer (glimage->texture,
                                 image->data.system_buffer.system_buffer,
                                 image->data.system_buffer.format,
                                 image->data.system_buffer.width,
                                 image->data.system_buffer.height);
  pgm_texture_set_matrix (glimage->texture, image->mapping_matrix);

  GST_OBJECT_UNLOCK (image);

  glimage->empty = FALSE;

  update_image_ratio (glimage);
  update_layout (glimage);
  update_slaves (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_GEN_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_from_image (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmImage *image = PGM_IMAGE (gldrawable->drawable);
  PgmGlViewport *glviewport = gldrawable->glviewport;
  PgmGlImage *master;

  GST_DEBUG_OBJECT (glimage, "set_from_image");

  /* Retrieve the master glimage */
  GST_OBJECT_LOCK (image);

  if (G_UNLIKELY (image->storage_type != PGM_IMAGE_IMAGE))
    {
      GST_OBJECT_UNLOCK (image);
      return;
    }

  GST_OBJECT_LOCK (glviewport);
  master = g_hash_table_lookup (glviewport->drawable_hash, image->master);
  GST_OBJECT_UNLOCK (glviewport);
  GST_OBJECT_UNLOCK (image);

  /* And use its texture */
  if (master)
    {
      glimage->texture = master->texture;
      glimage->empty = FALSE;
      update_image_ratio (glimage);
      update_layout (glimage);
    }
}

void
pgm_gl_image_set_mapping_matrix (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_mapping_matrix");

  update_mapping_matrix (glimage);
}

void
pgm_gl_image_set_alignment (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_alignment");

  update_alignment (glimage);
  update_layout (glimage);
}

void
pgm_gl_image_set_layout (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_layout");

  update_layout (glimage);
}

void
pgm_gl_image_set_interp (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_interp");

  update_interp (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_UPDATE_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_wrapping (PgmGlImage *glimage)
{
  PgmGlDrawable *gldrawable = PGM_GL_DRAWABLE (glimage);
  PgmContextTask *task;

  GST_DEBUG_OBJECT (glimage, "set_wrapping");

  update_wrapping (glimage);

  task = pgm_context_task_new (PGM_CONTEXT_UPDATE_TEXTURE, glimage->texture);
  pgm_context_push_immediate_task (gldrawable->glviewport->context, task);
}

void
pgm_gl_image_set_aspect_ratio (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_aspect_ratio");

  update_image_ratio (glimage);
  update_layout (glimage);
}

void
pgm_gl_image_set_border_width (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_border_width");

  update_border_width (glimage);
  update_layout (glimage);
}

void
pgm_gl_image_set_border_inner_color (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_border_inner_color");

  update_border_inner_color (glimage);
}

void
pgm_gl_image_set_border_outer_color (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "set_border_outer_color");

  update_border_outer_color (glimage);
}

void
pgm_gl_image_system_buffer_changed (PgmGlImage *glimage)
{
  GST_DEBUG_OBJECT (glimage, "system_buffer_changed");
}
