/****************************************************************************

 * Copyright (c) 2008-2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

File Name:          PowerSchemeMgmtProxy.h

Description:

    This file implements class PowerSchemeMgmtProxy.

Environment (opt):

    OS: Ubuntu
    SE: Code::Block

Notes (opt):

  =====================================================================

  Revision   Revision History               Author     Date

  =====================================================================

  0.1        Create                         Deng Jing 2008-7-20

  =====================================================================

****************************************************************************/

#include "include/PowerSchemeMgmtProxy.h"
#include "include/BatteryInfoMgmtProxy.h"
#include "include/PMUtility.h"
#include "include/DialogRegister.h"

#include <wx/utils.h>
#include <gdk/gdkx.h>
#include <X11/extensions/scrnsaver.h>
#include <dbus/dbus-glib.h>

#include <wx/dialog.h>

const char COMMON_MAX[] = "maxlevel";

const char* EFFECTIVE_ITEMS[EFFECTIVE_ITEMS_COUNT] = {"LCDBrightness","LANSpeed"};

unsigned long PowerSchemeMgmtProxy::g_CloseLCDTime = 0;
unsigned long PowerSchemeMgmtProxy::g_SuspendTime = 0;

int PowerSchemeMgmtProxy::g_LanOvertime = 0;

PowerSchemeMgmtProxy* PowerSchemeMgmtProxy::m_pPowerSchemeMgmtProxy = NULL;

//constructor
PowerSchemeMgmtProxy::PowerSchemeMgmtProxy()
{
    m_pPowerSchemeMgmtProxy = NULL;
    m_hasXSCExtension = 0;

    // Modified by Deng Jing on 04/10/2009
    m_CommonFilePath = PMUtility::WxStringToString(GLOBAL_CONFIG_DIR + LANG_DEFAULT_EN + COMMON_FILENAME);
    m_DefaultSchemeFilePath = PMUtility::WxStringToString(GLOBAL_CONFIG_DIR + LANG_DEFAULT_EN + SCHEME_FILENAME);
    m_HelpFilePath = HELP_DIR + LANG_DEFAULT_EN + HELP_FILENAME;
}

//destructor
PowerSchemeMgmtProxy::~PowerSchemeMgmtProxy()
{
}

//get the sington instance
PowerSchemeMgmtProxy* PowerSchemeMgmtProxy::GetInstance()
{
    if (m_pPowerSchemeMgmtProxy == NULL)
    {
        m_pPowerSchemeMgmtProxy = new PowerSchemeMgmtProxy();
        m_pPowerSchemeMgmtProxy->ReadWeight();
    }

    return m_pPowerSchemeMgmtProxy;
}

//get the active scheme ID
int PowerSchemeMgmtProxy::GetActivePwrScheme(int &schemeId)
{
    schemeId = 0;

    int ret = DCS_GetActivePwrScheme(&schemeId);

    return ret;
}

//add a scheme
int PowerSchemeMgmtProxy::AddPwrScheme(int &schemeId, PScheme_Info scheme)
{
    if ( (NULL == scheme) || (NULL == scheme->ACSettings) || (NULL == scheme->DCSettings) )
    {
        return FAILED;
    }

    char *pcSchemeName = const_cast<char*>((scheme->SchemeName).c_str());

    if ( SUCCESS != DCS_AddPwrScheme(pcSchemeName, pcSchemeName, 0, &schemeId) )
    {
        return FAILED;
    }

    // Set AC settings value
    if ( SUCCESS != EditPwrSchemeSettings(schemeId, 1, scheme->ACSettings) )
    {
        return FAILED;
    }

    // Set DC settings value
    if ( SUCCESS != EditPwrSchemeSettings(schemeId, 0, scheme->DCSettings) )
    {
        return FAILED;
    }

    return SUCCESS;
}

//apply the scheme settings
int PowerSchemeMgmtProxy::ApplySchemeSettings(int schemeId)
{
    int value = 0;

    //Get power type
    int isAC = BatteryInfoMgmtProxy::PM_GetACAdapterStatus();

    if (isAC != 1)
    {
        isAC = 0;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, CLOSE_LCD, &value) )
    {
        return FAILED;
    }

    g_CloseLCDTime = TIMELIST_SECOND[value];

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, STANDBY, &value) )
    {
        return FAILED;
    }

    g_SuspendTime = TIMELIST_SECOND[value];

    if (g_SuspendTime != 0  && g_CloseLCDTime >= g_SuspendTime)
    {
        g_CloseLCDTime = 0;
    }

    int retValue = 0;

    // Apply settings' value
    if (SUCCESS != DCS_GetPwrVariable(schemeId, isAC, LCD_BRIGHTNESS, &value))
    {
        return FAILED;
    }

    int iRet =  SUCCESS;

    iRet = DCS_SetLCDBrightness(value);

    // Apply Lan status
    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, LAN_STATUS, &value) )
    {
        return FAILED;
    }

    //iRet = DCS_SetLANStatus(value);

    if (value == 0)
    {
        g_LanOvertime = 1;
        DCS_Down_Interface_Request(0);
        DialogRegister::GetInstance()->GetPtrTask()->StartLANTimer();
    }
    else
    {
        iRet = DCS_SetLANStatus(value);
        DCS_GetLANStatus(&retValue);
        if (1 == retValue)
        {
            if ( FAILED == DCS_GetPwrVariable(schemeId, isAC, LAN_SPEED, &value) )
            {
                return FAILED;
            }

            iRet = DCS_SetLANSpeed((DCS_Lan_Speed)value);
        }
    }



    // Apply wireless status
    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, WLAN_STATUS, &value) )
    {
        return FAILED;
    }

    if (0 == value)
    {
        g_LanOvertime = 1;
        DCS_Down_Interface_Request(1);
        DialogRegister::GetInstance()->GetPtrTask()->StartLANTimer();
    }
    else
    {
        iRet = DCS_SetWirelessStatus(value);
    }

    return iRet;
}

//apply one scheme
int PowerSchemeMgmtProxy::ApplyPwrScheme(int schemeId)
{
    int iRet = SUCCESS;

    if (g_LanOvertime == 1)
    {
        return FAILED;
    }

    // Write configuration file to save the active power scheme id.
    iRet = DCS_SetActivePwrScheme(schemeId);
    if ( SUCCESS != iRet)
    {
        return APPLY_FAILED;
    }

    iRet = ApplySchemeSettings(schemeId);

    return iRet;
}

//delete one scheme
int PowerSchemeMgmtProxy::DeletePwrScheme(int schemeId)
{

    int ret = DCS_DeletePwrScheme(schemeId);
    return ret;
}

//edit the scheme
int PowerSchemeMgmtProxy::EditPwrSchemeSettings(int schemeId, int isAC, PScheme_Settings schemeSettings)
{
    if ( NULL == schemeSettings )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, LCD_BRIGHTNESS, schemeSettings->LCDBrightness))
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, LAN_SPEED, schemeSettings->LANSpeed) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, CLOSE_LCD, schemeSettings->CloseLCD) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, STANDBY, schemeSettings->Standby) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, LAN_STATUS, schemeSettings->LANStatus) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, WLAN_STATUS, schemeSettings->WLANStatus) )
    {
        return FAILED;
    }

    return SUCCESS;
}

//edit one scheme
int PowerSchemeMgmtProxy::EditPwrScheme(PScheme_Info scheme)
{

    if ( (NULL == scheme) || (NULL == scheme->ACSettings) || (NULL == scheme->DCSettings) )
    {
        return FAILED;
    }

    int schemeId = scheme->SchemeId;

    if ( SUCCESS != EditPowerSchemeName(scheme->SchemeName, schemeId))
    {
        return FAILED;
    }

    // Set AC settings value
    if ( FAILED == EditPwrSchemeSettings(schemeId, 1, scheme->ACSettings) )
    {
        return FAILED;
    }

    // Set DC settings value
    if ( FAILED == EditPwrSchemeSettings(schemeId, 0, scheme->DCSettings) )
    {
        return FAILED;
    }

    return SUCCESS;
}

//query scheme's settings
int PowerSchemeMgmtProxy::QueryPwrSchemeSettings(int schemeId, int isAC, PScheme_Settings schemeSettings)
{
    if( NULL == schemeSettings )
    {
        schemeSettings = new Scheme_Settings;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, LCD_BRIGHTNESS, &(schemeSettings->LCDBrightness)) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, LAN_SPEED, &(schemeSettings->LANSpeed)) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, CLOSE_LCD, &(schemeSettings->CloseLCD)) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, STANDBY, &(schemeSettings->Standby)) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, LAN_STATUS, &(schemeSettings->LANStatus)) )
    {
        return FAILED;
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, WLAN_STATUS, &(schemeSettings->WLANStatus)) )
    {
        return FAILED;
    }

    return SUCCESS;
}

//get one scheme settings
int PowerSchemeMgmtProxy::QueryPwrScheme(PScheme_Info scheme)
{
    int schemeId = scheme->SchemeId;
//    wxMessageBox(wxString::Format(wxT("scheme id is %d"), scheme->SchemeId));

    // Get name and default attribute
    if ( FAILED == GetPowerDefaultAttribute(schemeId, scheme->IsDefault))
    {
        return FAILED;
    }

    char nameBuf[STRING_MAX_LEN] = {'\0'};
    char descBuf[STRING_MAX_LEN] = {'\0'};

    if ( SUCCESS != DCS_GetPwrSchemeNameDesc(schemeId, nameBuf, STRING_MAX_LEN, descBuf, STRING_MAX_LEN) )
    {
        return FAILED;
    }

    scheme->SchemeName = nameBuf;

//    wxMessageBox(PMUtility::StringToWxString(nameBuf));

    // Get AC settings value
    PScheme_Settings schemeACSettings = new Scheme_Settings;
    if ( NULL == schemeACSettings )
    {
        return FAILED;
    }

    PScheme_Settings schemeDCSettings = new Scheme_Settings;
    if ( NULL == schemeDCSettings )
    {
        return FAILED;
    }

    if ( FAILED == QueryPwrSchemeSettings(schemeId, 1, schemeACSettings) ||
        NULL == schemeACSettings )
    {
        return FAILED;
    }

    // Get DC settings value
    if ( FAILED == QueryPwrSchemeSettings(schemeId, 0, schemeDCSettings) ||
        NULL == schemeDCSettings )
    {
        return FAILED;
    }

    scheme->ACSettings = schemeACSettings;
    scheme->DCSettings = schemeDCSettings;


    return SUCCESS;

}

//get one item of a scheme
int PowerSchemeMgmtProxy::GetPwrVariable(int schemeId, BOOL isAC, const char *pVarName, int &value)
{
    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, pVarName, &value) )
    {
        return FAILED;
    }

    return SUCCESS;
}

//set one item of a scheme
int PowerSchemeMgmtProxy::SetPwrVariable(int schemeId, BOOL isAC, const char *pVarName, int value)
{

    if ( SUCCESS != DCS_SetPwrVariable(schemeId, isAC, pVarName, value) )
    {
        return FAILED;
    }

    return SUCCESS;
}

//get disabled device list
int PowerSchemeMgmtProxy::GetDisabledDevices(int schemeId, int isAC, vector<string> &devicesVec)
{
    int value = 0;
    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, WLAN_STATUS, &value) )
    {
        return FAILED;
    }

    if ( 0 == value )
    {
        devicesVec.push_back(WLAN_STATUS);
    }

    if ( SUCCESS != DCS_GetPwrVariable(schemeId, isAC, LAN_STATUS, &value) )
    {
        return FAILED;
    }

    if ( 0 == value )
    {
        devicesVec.push_back(LAN_STATUS);
    }

    return SUCCESS;
}

//set disabled device
int PowerSchemeMgmtProxy::SetDisabledDevices(int schemeId, int isAC, vector<string> &devicesVec, int value)
{
    for (vector<string>::size_type ix = 0; ix != devicesVec.size(); ++ix)
    {
        string varName = devicesVec[ix];

        DCS_SetPwrVariable(schemeId, isAC, varName.c_str(), value);
    }

    return SUCCESS;
}

//get comboBox options
int PowerSchemeMgmtProxy::GetSettingItems(SETTING_ITEM_MAPPING &settingItemsMapping)
{
    TiXmlDocument xmlDoc(m_CommonFilePath.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlElement* xmlSubElement = 0;
    TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS);
    if (NULL == pNode)
	{
	    return FAILED;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_ITEMS);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlFirstSubElement = pNode->ToElement();

    TiXmlNode * child = 0;
    string varName;
    child = xmlFirstSubElement->FirstChild();

    while(NULL != child )
    {
        varName = child->ValueStr();

        xmlSubElement = child->ToElement();
        if (NULL == xmlSubElement)
        {
            return FAILED;
        }

        TiXmlElement* xmlItemElement = 0;
        SETTING_ITEM_VECTOR items;

        pNode = xmlSubElement->FirstChild(SCHEME_ITEM.c_str());

        while(pNode)
        {
            xmlItemElement = pNode->ToElement();

            Setting_Item_Info settingInfo;
            settingInfo.Name = varName;
            settingInfo.Description = xmlItemElement->Attribute(DESC_ATTRIBUTE.c_str());
            xmlItemElement->Attribute(VALUE_ATTRIBUTE.c_str(), &(settingInfo.Value));
            items.push_back(settingInfo);

            pNode = pNode->NextSibling();
        }

        settingItemsMapping.insert(make_pair(varName, items));

        child = child->NextSibling();
    }

    return SUCCESS;
}

///
// Get default name list of personal power scheme
// @parm: nameVec, name list
///
int PowerSchemeMgmtProxy::GetPersonalSchemeDefaultName(vector<string> &nameVec)
{
    TiXmlDocument xmlDoc(m_CommonFilePath.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlElement* xmlSubElement = 0;
    TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS);
    if (NULL == pNode)
	{
	    return FAILED;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_PERSONAL);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlFirstSubElement = pNode->ToElement();

    TiXmlNode * child = 0;
    string schemeName;
    child = xmlFirstSubElement->FirstChild(SCHEME_ITEM.c_str());

    while(NULL != child )
    {
        xmlSubElement = child->ToElement();
        if (NULL == xmlSubElement)
        {
            return FAILED;
        }

        schemeName = xmlSubElement->Attribute(DESC_ATTRIBUTE.c_str());
        nameVec.push_back(schemeName);

        child = child->NextSibling();
    }

    return SUCCESS;
}

///
// Get max number of power schemes allowed.
///
int PowerSchemeMgmtProxy::GetMaxNumberOfSchemes(int &maxNum)
{
    TiXmlDocument xmlDoc(m_CommonFilePath.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS);
    if (NULL == pNode)
	{
	    return FAILED;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_PERSONAL);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlFirstSubElement = pNode->ToElement();

	xmlFirstSubElement->Attribute(MAX_ATTRIBUTE.c_str(), &maxNum);

	return SUCCESS;
}

///
// Get description of devices to show in UI
///
int PowerSchemeMgmtProxy::GetDevicesDesc(map<string, string> &deviceDescMap)
{
    TiXmlDocument xmlDoc(m_CommonFilePath.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlElement* xmlSubElement = 0;
    TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS);
    if (NULL == pNode)
	{
	    return FAILED;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_DEVICES);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlFirstSubElement = pNode->ToElement();

    TiXmlNode * child = 0;
    string varName;
    string varDesc;
    child = xmlFirstSubElement->FirstChild(SCHEME_ITEM.c_str());

    while(NULL != child )
    {
        xmlSubElement = child->ToElement();
        if (NULL == xmlSubElement)
        {
            return FAILED;
        }

        varName = xmlSubElement->Attribute(NAME_ATTRIBUTE.c_str());
        varDesc = xmlSubElement->Attribute(DESC_ATTRIBUTE.c_str());
        deviceDescMap.insert(make_pair(varName, varDesc));

        child = child->NextSibling();
    }

    return SUCCESS;
}

//get the weight for effectiveness
int PowerSchemeMgmtProxy::ReadWeight()
{
    TiXmlDocument xmlDoc(m_CommonFilePath.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlElement* xmlSubElement = 0;
	TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS);
    if (NULL == pNode)
	{
	    return FAILED;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_EFFECTIVE);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlFirstSubElement = pNode->ToElement();

    TiXmlNode * child = 0;
    string varName;
    child = xmlFirstSubElement->FirstChild();

    while(NULL != child )
    {
        varName = child->ValueStr();

        xmlSubElement = child->ToElement();
        if (NULL == xmlSubElement)
        {
            return FAILED;
        }

        int value = 0;

        xmlSubElement->Attribute(MAX_ATTRIBUTE.c_str(), &value);
        m_EffectMaxValueMap.insert(make_pair(varName, value));

        string itemName;
        STRING_MAP itemMap;
        TiXmlElement* xmlItemElement = 0;

        pNode = xmlSubElement->FirstChild();

        while(pNode)
        {
            itemName = pNode->ValueStr();

            xmlItemElement = pNode->ToElement();

            xmlItemElement->Attribute(VALUE_ATTRIBUTE.c_str(), &value);
            itemMap.insert(make_pair(itemName, value));

            pNode = pNode->NextSibling();
        }

        m_EffectivenessMap.insert(make_pair(varName, itemMap));

        child = child->NextSibling();
    }

    return SUCCESS;
}

//calculate the effectiveness
int PowerSchemeMgmtProxy::CalculateEffectiveness(map<string, int>& valueMap, string varName)
{
    int percent = -1;
    STRING_MAP tempMap;
    map<string, STRING_MAP>::iterator mapIt = m_EffectivenessMap.find(varName);

    if ( mapIt != m_EffectivenessMap.end() )
    {
        tempMap = mapIt->second;

        int all = 0;
        int maxValue = 0;
        int minValue = 0;


        map<string, int>::const_iterator valueMapIt = valueMap.begin();
        while ( valueMapIt != valueMap.end() )
        {
            all += ((valueMapIt->second) *  tempMap[valueMapIt->first]);

            ++valueMapIt;
        }

        CalculateEffectiveRange(varName, maxValue, minValue);
        percent = (all - minValue) * 100 / (maxValue - minValue);

    }

    return percent;
}

//calculate performance
int PowerSchemeMgmtProxy::CalculatePerformance(map<string, int>& valueMap, int &persent)
{
    persent = CalculateEffectiveness(valueMap, PERFORMANCE);
    if (persent < 0)
    {
        return FAILED;
    }

    return SUCCESS;
}

//calculate display brightness
int PowerSchemeMgmtProxy::CalculateDisplayBrightness(map<string, int>& valueMap, int &persent)
{
    persent = CalculateEffectiveness(valueMap, DISPLAY);
    if (persent < 0)
    {
        return FAILED;
    }

    return SUCCESS;

}

///
// Calculate effectiveness of Battery Life
// @parm: valueMap, settings value
// @parm: percent, effectiveness percent
///
int PowerSchemeMgmtProxy::CalculateBatteryLife(map<string, int>& valueMap, int &persent)
{
    persent = CalculateEffectiveness(valueMap, BATTERYLIFE);
    if (persent < 0)
    {
        return FAILED;
    }

    return SUCCESS;
}

//get default scheme
int PowerSchemeMgmtProxy::ResetDefaultScheme(int schemeId, int isAC, PScheme_Settings schemeSettings)
{
    if ( NULL == schemeSettings )
    {
        schemeSettings = new Scheme_Settings;
    }

    map<string, int> defaultMap;

    string varName;

    GetDefaultSchemeInfo(schemeId, isAC, defaultMap);

    map<string, int>::const_iterator mapIt = defaultMap.begin();
    while ( mapIt != defaultMap.end() )
    {
        varName = mapIt->first;

        if ( 0 == varName.compare(LCD_BRIGHTNESS) )
        {
            schemeSettings->LCDBrightness = defaultMap[varName];
        }

        if ( 0 == varName.compare(LAN_SPEED) )
        {
            schemeSettings->LANSpeed = defaultMap[varName];
        }

        if ( 0 == varName.compare(CLOSE_LCD) )
        {
            schemeSettings->CloseLCD = defaultMap[varName];
        }

        if ( 0 == varName.compare(STANDBY) )
        {
            schemeSettings->Standby = defaultMap[varName];
        }

        if ( 0 == varName.compare(LAN_STATUS) )
        {
            schemeSettings->LANStatus = defaultMap[varName];
        }

        if ( 0 == varName.compare(WLAN_STATUS) )
        {
            schemeSettings->WLANStatus = defaultMap[varName];
        }

        ++mapIt;
    }


    return FAILED;
}

//get default scheme info
int PowerSchemeMgmtProxy::GetDefaultSchemeInfo(int schemeId, int isAC, map<string, int> &settingMapping)
{
    TiXmlDocument defaultXmlDoc(m_DefaultSchemeFilePath.c_str());
    if ( !defaultXmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlSubElement = 0;
	TiXmlNode * pNode = NULL;

	pNode = defaultXmlDoc.FirstChild(SCHEMES);
	xmlRootElement = pNode->ToElement();
	if (NULL == xmlRootElement)
	{
	    return FAILED;
	}

    pNode = xmlRootElement->FirstChild(SCHEME);

    while(pNode)
    {
        xmlSubElement = pNode->ToElement();
        int idTemp = 0;
        xmlSubElement->Attribute(SCHEME_ID.c_str(), &idTemp);

        if ( idTemp == schemeId )
        {
            TiXmlElement* xmlItemElement = 0;
            TiXmlElement* xmlValueElement = 0;
            int acValue = 0;
            int dcValue = 0;
            string varName;
            TiXmlNode * child = 0;

            child = xmlSubElement->FirstChild();
            while(child)
            {
                varName = child->ValueStr();

                xmlItemElement = child->ToElement();
                if (NULL == xmlItemElement)
                {
                    return DCS_FAIL_OPERATION;
                }

                if (isAC)
                {
                    xmlValueElement = xmlItemElement->FirstChildElement(ACVALUE);
                    if (NULL == xmlValueElement)
                    {
                        return FAILED;
                    }

                    xmlValueElement->Attribute(VALUE_ATTRIBUTE.c_str(), &acValue);
                    settingMapping.insert(make_pair(varName, acValue));
                }
                else
                {
                    xmlValueElement = xmlItemElement->FirstChildElement(DCVALUE);
                    if (NULL == xmlValueElement)
                    {
                        return FAILED;
                    }
                    xmlValueElement->Attribute(VALUE_ATTRIBUTE.c_str(), &dcValue);
                    settingMapping.insert(make_pair(varName, dcValue));
                }

                child = child->NextSibling();
            }

            return SUCCESS;
        }

        pNode = pNode->NextSibling();
    }

    return FAILED;
}

//edit scheme name
int PowerSchemeMgmtProxy::EditPowerSchemeName(string schemeName, int schemeId)
{
    if (schemeName.empty())
    {
        return FAILED;
    }

    string homeDir = PMUtility::GetHomeDir();
    if (homeDir.empty())
    {
        return FAILED;
    }

    homeDir += PRIVATE_SCHEME_FILEPATH;

    TiXmlDocument xmlDoc(homeDir.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlSubElement = 0;
	TiXmlNode * pNode = NULL;

	pNode = xmlDoc.FirstChild(SCHEMES);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlRootElement = pNode->ToElement();

    pNode = xmlRootElement->FirstChild(SCHEME);
    while(pNode)
    {
        xmlSubElement = pNode->ToElement();
        int idTemp = 0;
        xmlSubElement->Attribute(SCHEME_ID.c_str(), &idTemp);

        if ( idTemp == schemeId )
        {
            xmlSubElement->SetAttribute(NAME_ATTRIBUTE.c_str(), schemeName);
            break;
        }

        pNode = pNode->NextSibling();
    }

    if ( xmlDoc.SaveFile() )
    {
        return SUCCESS;
    }

    return FAILED;
}

//get scheme is default
int PowerSchemeMgmtProxy::GetPowerDefaultAttribute(int schemeId, int &isDefault)
{
    isDefault = 0;

    string homeDir = PMUtility::GetHomeDir();
    if (homeDir.empty())
    {
        return FAILED;
    }

    homeDir += PRIVATE_SCHEME_FILEPATH;

    TiXmlDocument xmlDoc(homeDir.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlSubElement = 0;
	TiXmlNode * pNode = NULL;

	pNode = xmlDoc.FirstChild(SCHEMES);
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlRootElement = pNode->ToElement();

   pNode = xmlRootElement->FirstChild(SCHEME);
    int idTemp = 0;

    while(pNode)
    {
        xmlSubElement = pNode->ToElement();

        xmlSubElement->Attribute(SCHEME_ID.c_str(), &idTemp);

        if ( idTemp == schemeId )
        {
            xmlSubElement->Attribute(DEFAULT_ATTRIBUTE.c_str(), &isDefault);

            break;
        }

        pNode = pNode->NextSibling();
    }

	return SUCCESS;
}

//apply the active scheme at first
int PowerSchemeMgmtProxy::InitScheme()
{
    int dummy = 0;
	m_hasXSCExtension = XScreenSaverQueryExtension( GDK_DISPLAY(), &dummy, &dummy );
    int schemeId = 0;
    GetActivePwrScheme(schemeId);
    return ApplyPwrScheme(schemeId);
}

//get system idle time
unsigned long PowerSchemeMgmtProxy::GetIdleTime()
{
	if(m_hasXSCExtension)
	{
		XScreenSaverInfo* mitInfo = 0;
		if (!mitInfo)
		{
		    mitInfo = XScreenSaverAllocInfo();
		}

		XScreenSaverQueryInfo(GDK_DISPLAY(), DefaultRootWindow (GDK_DISPLAY()), mitInfo);
		m_idleTime = mitInfo->idle;
	}
	else
	{
		m_idleTime = 0;
	}

	return m_idleTime;
}

#define	GPM_DBUS_SERVICE		"org.freedesktop.PowerManagement"
#define	GPM_DBUS_INHIBIT_PATH		"/org/freedesktop/PowerManagement"
#define	GPM_DBUS_INHIBIT_INTERFACE	"org.freedesktop.PowerManagement"

#define GS_LISTENER_SERVICE	"org.gnome.ScreenSaver"
#define GS_LISTENER_PATH	"/org/gnome/ScreenSaver"
#define GS_LISTENER_INTERFACE	"org.gnome.ScreenSaver"

//suspend event handler
bool PowerSchemeMgmtProxy::EnterSuspend()
{
	gboolean         res;
	gboolean ret;
	gboolean temp = TRUE;
	guint sleepcount = 0;
	GError          *error = NULL;
	DBusGProxy      *proxy = NULL;
	DBusGConnection *session_connection = NULL;
	DBusGProxy      *screenproxy = NULL;

	/* get the DBUS session connection */
	session_connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
	if (error) {
		g_warning ("DBUS cannot connect : %s", error->message);
		g_error_free (error);
		return false;
	}
    screenproxy = dbus_g_proxy_new_for_name (session_connection,
                GS_LISTENER_SERVICE,
				  GS_LISTENER_PATH,
				  GS_LISTENER_INTERFACE);

    dbus_g_proxy_call_no_reply (screenproxy, "Lock", G_TYPE_INVALID);

    while (!(ret = dbus_g_proxy_call (screenproxy, "GetActive", &error,
				 G_TYPE_INVALID,
				 G_TYPE_BOOLEAN, &temp,
				 G_TYPE_INVALID)))
    {
        g_usleep (1000 * 100);
		if (sleepcount++ > 50) {
			break;
		}
    }

    // modified by dengjing on 2009/04/02
//    wxString home = ::wxGetHomeDir();
//    string tmp(home.mb_str(wxConvUTF8));

    string tmp("/tmp");

    char cmd[256] = {'\0'};
    sprintf(cmd, "touch %s/.suspend", tmp.c_str());
    system(cmd);

	/* get the proxy with g-p-m */
	proxy = dbus_g_proxy_new_for_name (session_connection,
					   GPM_DBUS_SERVICE,
					   GPM_DBUS_INHIBIT_PATH,
					   GPM_DBUS_INHIBIT_INTERFACE);
	if (proxy == NULL) {
		g_warning ("Could not get DBUS proxy: %s", GPM_DBUS_SERVICE);
		return false;
	}

	res = dbus_g_proxy_call (proxy,
				 "Suspend",
				 &error,
				 G_TYPE_INVALID,
				 G_TYPE_INVALID);

	/* check the return value */
	if (! res) {
		g_warning ("Suspend method failed");
		return false;
	}

	/* check the error value */
	if (error != NULL) {
		g_warning ("Suspend problem : %s", error->message);
		g_error_free (error);
		return false;
	}

    // modified by dengjing on 2009/04/02
    memset(cmd, 0, 256);
    sprintf(cmd, "touch %s/.resume", tmp.c_str());
    system(cmd);

    dbus_g_proxy_call_no_reply (screenproxy,
				    "SimulateUserActivity",
				    G_TYPE_INVALID);

	g_object_unref (G_OBJECT (proxy));
	g_object_unref (G_OBJECT (screenproxy));
    return true;

}

//get the max count of scheme
int PowerSchemeMgmtProxy::GetPwrVariableMaxCount(const char *itemName, int &count)
{
    TiXmlDocument xmlDoc(m_CommonFilePath.c_str());
    if ( !xmlDoc.LoadFile() )
    {
        return FAILED;
    }

    TiXmlElement* xmlRootElement = 0;
    TiXmlElement* xmlFirstSubElement = 0;
    TiXmlElement* xmlSubElement = 0;
    TiXmlNode * pNode = 0;

    pNode = xmlDoc.FirstChild(COMMON_SETTINGS.c_str());
    if (NULL == pNode)
	{
	    return FAILED;
	}
    xmlRootElement = pNode->ToElement();

	pNode = xmlRootElement->FirstChild(COMMON_ITEMS.c_str());
	if (NULL == pNode)
	{
	    return FAILED;
	}
	xmlFirstSubElement = pNode->ToElement();

    pNode = xmlFirstSubElement->FirstChild(itemName);
    if (NULL == pNode)
    {
        return FAILED;
    }
    xmlSubElement = pNode->ToElement();

    xmlSubElement->Attribute(COMMON_MAX, &count);

    return SUCCESS;
}

//calculate the effective range
int PowerSchemeMgmtProxy::CalculateEffectiveRange(string effectiveName, int &maxRange, int &minRange)
{
    map<string, STRING_MAP>::iterator mapIt = m_EffectivenessMap.find(effectiveName);
    maxRange = minRange = 0;

    int count = 0;
    STRING_MAP tempMap;
    if ( mapIt != m_EffectivenessMap.end() )
    {
        tempMap = mapIt->second;

        for (int i = 0; i < EFFECTIVE_ITEMS_COUNT; i++)
        {
            string tempItem = EFFECTIVE_ITEMS[i];
            GetPwrVariableMaxCount(tempItem.c_str(), count);
            if (tempMap[tempItem] >= 0)
            {
                maxRange += count * tempMap[tempItem];
                minRange -= tempMap[tempItem];
            }
            else
            {
                maxRange += 0;
                minRange += (count + 1) * tempMap[tempItem];
            }
        }
    }

     return SUCCESS;
}

// Initialize common file path
void PowerSchemeMgmtProxy::InitCommonResourceFilePath(wxString language)
{
    m_CommonFilePath = PMUtility::WxStringToString(GLOBAL_CONFIG_DIR + language + COMMON_FILENAME);

}

// Inintialize default scheme file path
void PowerSchemeMgmtProxy::InitDefaultResourceFilePath(wxString language)
{
    m_DefaultSchemeFilePath = PMUtility::WxStringToString(GLOBAL_CONFIG_DIR + language + SCHEME_FILENAME);

}

// Initialize help file path
void PowerSchemeMgmtProxy::InitHelpResourceFilePath(wxString language)
{
     m_HelpFilePath = HELP_DIR + language + HELP_FILENAME;
}

// Support common file path
void PowerSchemeMgmtProxy::GetCommonResourceFilePath(string& filePath)
{
    filePath = m_CommonFilePath;
}

// Support default scheme file path
void PowerSchemeMgmtProxy::GetDefaultResourceFilePath(string& filePath)
{
    filePath = m_DefaultSchemeFilePath;
}

// Support help file path
void PowerSchemeMgmtProxy::GetHelpResourceFilePath(wxString& filePath)
{
    filePath = m_HelpFilePath;
}
