/*
 * This file is part of signon-plugin-google
 *
 * Copyright (C) 2011 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "client-login-data.h"
#include "plugin.h"

#include <QUrl>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QNetworkProxy>
#include <QSslError>

#ifdef TRACE
#undef TRACE
#endif
#define TRACE() qDebug() << __FILE__ << __LINE__ << __func__ << ":"
#define S(string)   QLatin1String(string)

using namespace SignOn;
using namespace GooglePlugin;

namespace GooglePlugin {

static const QString CLIENT_LOGIN = S("ClientLogin");
static const QString HOSTED_OR_GOOGLE = S("HOSTED_OR_GOOGLE");
static const QString DEFAULT_SERVICE = S("xapi");
static const QString ACTION_URL =
    S("https://www.google.com/accounts/ClientLogin");
static const QString CAPTCHA_URL = S("https://www.google.com/accounts/");
static const QString UNLOCK_URL =
    S("https://www.google.com/accounts/DisplayUnlockCaptcha");

typedef QMap<QByteArray,QByteArray> StringMap;

class PluginPrivate: public QObject
{
    Q_OBJECT
    Q_DECLARE_PUBLIC(Plugin)

public:
    PluginPrivate(Plugin *plugin):
        QObject(plugin),
        q_ptr(plugin),
        m_networkAccessManager(new QNetworkAccessManager(this))
    {
    }

    ~PluginPrivate()
    {
        TRACE();
    }

    void cancel() {
        delete m_networkAccessManager;
        m_networkAccessManager = new QNetworkAccessManager(this);
    }

    StringMap parseReply(QNetworkReply *reply);
    bool handleError(const QNetworkReply *reply);
    QNetworkReply *request(QNetworkAccessManager::Operation operation,
                           const QString &baseUrl,
                           const QByteArray &queryData);
    void clientLogin(const ClientLoginData &data);
    QVariantMap clientLoginHeaders(const ClientLoginData &data);
    QByteArray encodeParams(const QVariantMap &params);
    bool validateInput(const SignOn::SessionData &inData,
                       const QString &mechanism);
    void onClientLoginFinished(QNetworkReply *reply);
    void handleClientLoginError(const StringMap &replyData);

private Q_SLOTS:
    void onClientLoginFinished();
    void onClientLoginError(QNetworkReply::NetworkError error);
    void onClientLoginSslErrors(QList<QSslError> sslErrors);

private:
    mutable Plugin *q_ptr;
    ClientLoginData m_sessionData;
    QNetworkAccessManager *m_networkAccessManager;
    QNetworkProxy m_networkProxy;
    Error m_lastError;
    QString m_proxyUrl;
};

} // namespace

StringMap PluginPrivate::parseReply(QNetworkReply *reply)
{
    QByteArray replyData = reply->readAll();
    QList<QByteArray> lines = replyData.split('\n');

    StringMap ret;
    foreach (QByteArray line, lines) {
        int equal = line.indexOf('=');
        if (equal == -1) continue; // ignore this line

        QByteArray key = line.left(equal);
        QByteArray value = line.mid(equal + 1);
        ret.insert(key, value);
    }
    return ret;
}

bool PluginPrivate::handleError(const QNetworkReply *reply)
{
    Q_Q(Plugin);

    switch (reply->error()) {
    case QNetworkReply::SslHandshakeFailedError:
        // handled by SSL error handler; fall through
    case QNetworkReply::NoError:
    case QNetworkReply::ContentOperationNotPermittedError:
        return false;

    default:
        Q_EMIT q->error(Error(Error::Network, reply->errorString()));
        return true;
    }
}

QNetworkReply *
PluginPrivate::request(QNetworkAccessManager::Operation operation,
                       const QString &baseUrl,
                       const QByteArray &queryData)
{
    QNetworkReply *reply;
    QUrl url(baseUrl);

    m_networkAccessManager->setProxy(m_networkProxy);

    if (operation == QNetworkAccessManager::GetOperation) {
        url.setEncodedQuery(queryData);

        QNetworkRequest request(url);
        reply = m_networkAccessManager->get(request);
    } else {
        QNetworkRequest request(url);
        request.setRawHeader("ContentType",
                             "application/x-www-form-urlencoded");
        TRACE() << "Query data" << queryData;
        reply = m_networkAccessManager->post(request, queryData);
    }

    TRACE() << "Requested url: " << url;

    return reply;
}

void PluginPrivate::clientLogin(const ClientLoginData &data)
{
    QString baseUrl = data.ActionUrl().isEmpty() ?
        ACTION_URL : data.ActionUrl();

    QNetworkReply *reply = request(QNetworkAccessManager::PostOperation,
                                   baseUrl,
                                   encodeParams(clientLoginHeaders(data)));
    QObject::connect(reply, SIGNAL(finished()),
                     this, SLOT(onClientLoginFinished()));
    QObject::connect(reply, SIGNAL(error(QNetworkReply::NetworkError)),
                     this,
                     SLOT(onClientLoginError(QNetworkReply::NetworkError)));
    QObject::connect(reply, SIGNAL(sslErrors(QList<QSslError>)),
                     this, SLOT(onClientLoginSslErrors(QList<QSslError>)));
    if (reply->isFinished()) {
        onClientLoginFinished(reply);
    }
}

QVariantMap PluginPrivate::clientLoginHeaders(const ClientLoginData &data)
{
    QVariantMap headers;

    QString value = data.AccountType();
    headers.insert(S("accountType"),
                   value.isEmpty() ? HOSTED_OR_GOOGLE : value);

    headers.insert(S("Email"), data.UserName());
    headers.insert(S("Passwd"), data.Secret());

    headers.insert(S("service"),
                   data.Service().isEmpty() ? DEFAULT_SERVICE : data.Service());
    headers.insert(S("source"), data.Source());

    if (!data.LoginToken().isEmpty())
        headers.insert(S("logintoken"), data.LoginToken());
    if (!data.LoginCaptcha().isEmpty())
        headers.insert(S("logincaptcha"), data.LoginCaptcha());

    return headers;
}

QByteArray PluginPrivate::encodeParams(const QVariantMap &params)
{
    QString queryString;
    QUrl url;

    QMapIterator<QString,QVariant> i(params);
    while (i.hasNext()) {
        i.next();
        url.addEncodedQueryItem(i.key().toUtf8(),
                                QUrl::toPercentEncoding(i.value().toString()));
    }

    return url.encodedQuery();
}

bool PluginPrivate::validateInput(const SignOn::SessionData &inData,
                                  const QString &mechanism)
{
    if (mechanism == CLIENT_LOGIN) {
        ClientLoginData input = inData.data<ClientLoginData>();
        if (input.Source().isEmpty()) {
            return false;
        }
    }

    return true;
}

void PluginPrivate::handleClientLoginError(const StringMap &replyData)
{
    Q_Q(Plugin);

    const QByteArray &error = replyData["Error"];

    if (error == "BadAuthentication") {
        SignOn::UiSessionData uiSession;
        if (m_sessionData.UserName().isEmpty()) {
            uiSession.setQueryUserName(true);
        } else {
            uiSession.setUserName(m_sessionData.UserName());
        }
        uiSession.setQueryPassword(true);
        uiSession.setQueryMessageId(QUERY_MESSAGE_NOT_AUTHORIZED);

        Q_EMIT q->userActionRequired(uiSession);
        m_lastError = Error::NotAuthorized;
    } else if (error == "NotVerified" ||
               error == "TermsNotAgreed") {
        if (replyData.contains("Url")) {
            SignOn::UiSessionData uiSession;
            QString url = QString::fromLatin1(replyData["Url"].constData());
            uiSession.setOpenUrl(url);
            uiSession.setNetworkProxy(m_proxyUrl);

            Q_EMIT q->userActionRequired(uiSession);
            m_lastError = Error::NotAuthorized;
        } else {
            Q_EMIT q->error(Error::NotAuthorized);
        }
    } else if (error == "CaptchaRequired") {
        SignOn::UiSessionData uiSession;
        if (m_sessionData.UserName().isEmpty()) {
            uiSession.setQueryUserName(true);
        } else {
            uiSession.setUserName(m_sessionData.UserName());
        }
        uiSession.setQueryPassword(true);
        QString captchaUrl =
            CAPTCHA_URL +
            QString::fromLatin1(replyData["CaptchaUrl"].constData());
        uiSession.setCaptchaUrl(captchaUrl);
        uiSession.setForgotPasswordUrl(UNLOCK_URL);
        uiSession.setNetworkProxy(m_proxyUrl);

        // remember the captcha token
        m_sessionData.setLoginToken(replyData["CaptchaToken"]);

        Q_EMIT q->userActionRequired(uiSession);
        m_lastError = Error::NotAuthorized;
    } else {
        Q_EMIT q->error(Error(Error::Unknown,
                              QString::fromLatin1(error.constData())));
    }
}

void PluginPrivate::onClientLoginFinished(QNetworkReply *reply)
{
    Q_Q(Plugin);

    Q_ASSERT(reply != 0);
    TRACE();

    reply->deleteLater();

    if (handleError(reply)) {
        // Some error occured
        return;
    }

    StringMap replyData = parseReply(reply);
    TRACE() << "Response: " << replyData;
    if (reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() ==
        200) {
        if (replyData.contains("Auth")) {
            ClientLoginTokenData response;
            response.setAuthToken(replyData["Auth"]);
            Q_EMIT q->result(response);
        } else {
            Q_EMIT q->error(Error(Error::Unknown,
                                  QString("Access token not present")));
        }
    } else if (replyData.contains("Error")) {
        handleClientLoginError(replyData);
    } else {
        Q_EMIT q->error(Error(Error::Unknown,
                              QString("Error reply with no error code")));
    }
}

void PluginPrivate::onClientLoginFinished()
{
    onClientLoginFinished(qobject_cast<QNetworkReply*>(sender()));
}

void PluginPrivate::onClientLoginError(QNetworkReply::NetworkError error)
{
    TRACE() << "Network error: " << error;
    // errors are handled in onClientLoginFinished
}

void PluginPrivate::onClientLoginSslErrors(QList<QSslError> errors)
{
    Q_Q(Plugin);

    TRACE() << "Errors: " << errors;
    QString message;
    foreach (QSslError error, errors) {
        message += error.errorString() + ";";
    }

    Q_EMIT q->error(Error(Error::Ssl, message));
}

Plugin::Plugin(QObject *parent):
    AuthPluginInterface(parent),
    d_ptr(new PluginPrivate(this))
{
    TRACE();
}

Plugin::~Plugin()
{
    TRACE();
    delete d_ptr;
    d_ptr = 0;
}

QString Plugin::type() const
{
    TRACE();
    return QString("google");
}

QStringList Plugin::mechanisms() const
{
    TRACE();
    QStringList ret;
    ret.append(CLIENT_LOGIN);
    return ret;
}

void Plugin::cancel()
{
    Q_D(Plugin);

    TRACE();
    d->cancel();
}

void Plugin::process(const SignOn::SessionData &inData,
                     const QString &mechanism)
{
    Q_D(Plugin);

    TRACE();

    if (!d->validateInput(inData, mechanism)) {
        TRACE() << "Invalid parameters passed";
        Q_EMIT error(Error::MissingData);
        return;
    }

    d->m_proxyUrl = inData.NetworkProxy();
    //set proxy from params
    if (!d->m_proxyUrl.isEmpty()) {
        QUrl url(d->m_proxyUrl);
        if (!url.host().isEmpty()) {
            d->m_networkProxy = QNetworkProxy(QNetworkProxy::HttpProxy,
                                              url.host(),
                                              url.port(),
                                              url.userName(),
                                              url.password());
            TRACE() << url.host() << ":" << url.port();
        }
    } else {
        d->m_networkProxy = QNetworkProxy::applicationProxy();
    }

    if (mechanism == CLIENT_LOGIN) {
        d->m_sessionData = inData.data<ClientLoginData>();

        /* if username or password are missing, invoke the UI */
        if (inData.UserName().isEmpty() || inData.Secret().isEmpty()) {
            // store the error, if the UI interaction fails
            d->m_lastError = Error(Error::InvalidCredentials);

            SignOn::UiSessionData uiSession = inData.data<UiSessionData>();
            // ask for username, if empty
            if (inData.UserName().isEmpty())
                uiSession.setQueryUserName(true);
            // always ask for password
            uiSession.setQueryPassword(true);
            uiSession.setQueryMessageId(QUERY_MESSAGE_LOGIN);
            Q_EMIT userActionRequired(uiSession);
            return;
        }

        d->clientLogin(d->m_sessionData);
    } else {
        Q_EMIT error(Error::MechanismNotAvailable);
    }
}

void Plugin::userActionFinished(const SignOn::UiSessionData &data)
{
    Q_D(Plugin);
    TRACE();

    if (data.QueryErrorCode() != QUERY_ERROR_NONE) {
        TRACE() << "userActionFinished with error: " << data.QueryErrorCode();
        if (data.QueryErrorCode() == QUERY_ERROR_CANCELED) {
            Q_EMIT error(Error(Error::SessionCanceled,
                               QLatin1String("Cancelled by user")));
        } else {
            Q_EMIT error(d->m_lastError);
        }
        return;
    }

    // fill in the provided data
    if (!data.UserName().isEmpty()) {
        d->m_sessionData.setUserName(data.UserName());
    }
    if (!data.Secret().isEmpty()) {
        d->m_sessionData.setSecret(data.Secret());
    }
    if (!data.CaptchaResponse().isEmpty()) {
        d->m_sessionData.setLoginCaptcha(data.CaptchaResponse());
    }

    // reattempt the authentication
    d->clientLogin(d->m_sessionData);
}

void Plugin::refresh(const SignOn::UiSessionData &data)
{
    TRACE();
    Q_EMIT refreshed(data);
}

SIGNON_DECL_AUTH_PLUGIN(GooglePlugin::Plugin)
#include "plugin.moc"
